import logging

from django.conf import settings
from fan.utils.logging_ import (
    LockedHandler,
    encode_complex_or_none,
    inject_timestamp_hack,
    DEFAULT_EXCLUDED_FIELDS,
    DEFAULT_ORDERED_FIRST,
    DEFAULT_RENAME_FIELDS,
    TSKVFormatter,
)
from logging.handlers import RotatingFileHandler
from traceback import extract_tb

_typed_log_handler = None


LOGGED_QUERY_PARAMS = (
    "user_id",
    "org_id",
    "account_slug",
    "campaign_slug",
)

LOGGED_HEADERS = (
    "x_request_id",
    "x_request_timeout",
    "x_request_attempt",
    "x_real_ip",
)


def update_custom_log_fields(request, fields):
    if not hasattr(request, "custom_log_fields"):
        request.custom_log_fields = {}
    request.custom_log_fields.update(fields)


def log_api_response(request, response, exc):
    data = {
        "status_code": response.status_code if response is not None else None,
        "request": request.get_full_path(),
        "exc_message": _get_exc_message(exc),
        "exc_detail": _get_exc_detail(exc),
    }
    for param_name in LOGGED_QUERY_PARAMS:
        data[param_name] = request.query_params.get(param_name, None)
    for header_name in LOGGED_HEADERS:
        data[header_name] = request.META.get("HTTP_" + header_name.upper(), None)
    _update_with_custom_fields(data, request)
    _log(data)


def _get_exc_message(exc):
    if exc is None:
        return None
    if hasattr(exc, "traceback"):
        filename, lineno, _, _ = extract_tb(exc.traceback)[-1]
        return "%s raised at %s:%s" % (type(exc).__name__, filename, lineno)
    else:
        return "%s raised" % type(exc).__name__


def _get_exc_detail(exc):
    if exc is not None and hasattr(exc, "detail"):
        return exc.detail
    return None


def _update_with_custom_fields(data, request):
    if not hasattr(request, "custom_log_fields"):
        return
    data.update(request.custom_log_fields)


def _log(data):
    flat_data = encode_complex_or_none(data.copy())
    flat_data = inject_timestamp_hack(flat_data)
    handler = _get_typed_log_handler()
    handler.emit(logging.makeLogRecord(flat_data))


def _get_typed_log_handler():
    global _typed_log_handler
    if _typed_log_handler is None:
        _typed_log_handler = _create_typed_log_handler(_create_base_log_handler())
    return _typed_log_handler


def _create_base_log_handler():
    return RotatingFileHandler(filename=settings.ACCESS_TYPEDLOG_FILENAME)


def _create_typed_log_handler(handler):
    fmt = {
        "tskv_format": settings.ACCESS_TYPEDLOG_TSKV_FORMAT,
        "exclude_fields": DEFAULT_EXCLUDED_FIELDS
        | {"threadName", "levelname", "message", "caller_info"},
        "ordered_first": DEFAULT_ORDERED_FIRST,
        "rename_fields": DEFAULT_RENAME_FIELDS,
    }
    handler.setFormatter(TSKVFormatter(**fmt))
    return LockedHandler(handler)
