from django.conf import settings
from django.db.models import Q
from django.utils.timezone import now as utcnow
from fan.accounts.organizations.users import is_member
from fan.accounts.organizations.limits import get_trusty as is_trusty_organization
from fan.accounts.organizations.settings import trusty
from fan.models.account import Account
from fan.models.user import UserRole


def get_account_by_name(name):
    return Account.objects.get(name=name)


def get_accounts_by_user_id(user_id):
    roles = UserRole.objects.filter(user_id=user_id)
    return [role.account for role in roles]


def get_accounts_by_org_id(org_id):
    return Account.objects.filter(org_id=org_id)


def get_account_users(account):
    user_roles = UserRole.objects.filter(account=account, role=UserRole.ROLES.USER)
    return [user_role.user_id for user_role in user_roles]


def user_has_role_in_account(user_id, account):
    if not _user_has_role_in_account(user_id, account):
        return False
    if _update_role_ts_if_outdated(user_id, account):
        _try_recheck_user_role(user_id, account)
    return _user_has_role_in_account(user_id, account)


def get_account_from_logins(account):
    return list(account.from_logins)


def get_trusty(account):
    value = trusty(account.org_id)
    if value is not None:
        return value
    return is_trusty_organization(account.org_id)


def need_to_silently_drop_sents(account):
    if not settings.SILENTLY_DROP_SENTS_FOR_NEW_UNTRUSTY_ACCOUNTS:
        return False
    return account.created_at >= settings.IS_NEW_ACCOUNT_AFTER and not get_trusty(account)


def _user_has_role_in_account(user_id, account):
    return UserRole.objects.filter(user_id=user_id, account=account).exists()


def _update_role_ts_if_outdated(user_id, account):
    last_checked_before = utcnow() - settings.RECHECK_USER_ROLE_MIN_PERIOD
    need_recheck = UserRole.objects.filter(
        Q(user_id=user_id),
        Q(account=account),
        Q(checked_at__lt=last_checked_before) | Q(checked_at__isnull=True),
    ).update(checked_at=utcnow())
    return need_recheck


def _try_recheck_user_role(user_id, account):
    try:
        if not is_member(user_id, account.org_id):
            UserRole.objects.filter(user_id=user_id, account=account).delete()
    except:
        pass
