from django.conf import settings
from fan.utils.tvm import TvmSession
from fan.utils.emails import get_domain


class CheckDomainResult:
    OK = "ok"
    NOT_BELONGS = "not_belongs"
    NO_MX = "no_mx"
    NO_DKIM = "no_dkim"
    NO_SPF = "no_spf"


def check_campaign_domain(campaign):
    domain = get_domain(campaign.from_email)
    res = domain_belongs_to_organization(domain, campaign.account.org_id)
    if res != CheckDomainResult.OK:
        return res
    res = check_domain_configured(domain)
    return res


def domain_belongs_to_organization(domain, org_id):
    if domain.lower() in _get_owned_domains(org_id):
        return CheckDomainResult.OK
    return CheckDomainResult.NOT_BELONGS


def get_organization_domains(org_id):
    domains = []
    for domain, details in list(_get_owned_domains(org_id).items()):
        try:
            domain_status = _get_domain_status(domain)
        except:
            domain_status = {}
        domains.append(
            {
                "domain": domain,
                "master": details["master"],
                "mx": domain_status.get("mx", None),
                "dkim": domain_status.get("dkim", None),
                "spf": domain_status.get("spf", None),
            }
        )
    return domains


def _get_owned_domains(org_id):
    resp = _request_domains(org_id)
    resp.raise_for_status()
    return {
        domain["name"]: {"master": domain["master"]}
        for domain in resp.json()
        if domain["owned"] == True
    }


def _request_domains(org_id):
    session = TvmSession(settings.DIRECTORY_TVM_ID)
    url = "{}/v11/domains/?fields=owned,name,master".format(settings.DIRECTORY_HOST)
    headers = {"X-ORG-ID": str(org_id)}
    return session.get(url, headers=headers, timeout=settings.DIRECTORY_TIMEOUT)


def check_domain_configured(domain):
    domain_status = _get_domain_status(domain)
    if not domain_status["mx"]:
        return CheckDomainResult.NO_MX
    if not domain_status["dkim"]:
        return CheckDomainResult.NO_DKIM
    if not domain_status["spf"]:
        return CheckDomainResult.NO_SPF
    return CheckDomainResult.OK


def _get_domain_status(domain):
    resp = _request_domain_status(domain.lower())
    resp.raise_for_status()
    response = _parse_domain_status_response(resp)
    return {
        "mx": _mx_configured(response),
        "dkim": _dkim_configured(response),
        "spf": _spf_configured(response),
    }


def _request_domain_status(domain):
    session = TvmSession(settings.GENDARME_TVM_ID)
    url = "{}/domain/status".format(settings.GENDARME_HOST)
    params = {"name": domain}
    return session.get(url, params=params, timeout=settings.GENDARME_TIMEOUT)


def _parse_domain_status_response(resp):
    json = resp.json()
    if json["status"] != "ok":
        raise Exception(
            'Gendarme domain status response contains unexpected status "%s"' % json["status"]
        )
    return json["response"]


def _mx_configured(response):
    return response["mx"]["match"] == True


def _dkim_configured(response):
    return any([dkim["match"] == True for dkim in response["dkim"]])


def _spf_configured(response):
    return response["spf"]["match"] == True
