from datetime import date, timedelta
from django.conf import settings
from django.db.models import Case, Sum, Q, Value, When
from django.db.models.functions import Coalesce
from fan.campaigns.get import get_draft_campaigns_count
from fan.db.decorator import atomic_auto_retry
from django.utils import timezone
from fan.delivery.status import DeliveryStatus
from fan.models import OrganizationSettings, Campaign, DeliveryErrorStats, EditorialLog, Maillist
from fan.accounts.organizations.settings import (
    campaigns_per_day_limit,
    draft_campaigns_limit,
    maillists_limit,
    test_sends_per_day_limit,
    send_emails_limit,
    trusty,
)


DAYS_IN_MONTH = timedelta(days=30)

QUERY_UPLOADED_EMAILS = Q(status__exact=DeliveryStatus.EMAIL_UPLOADED)
QUERY_FAILED_EMAILS = Q(status__in=DeliveryStatus.EMAIL_SEND_FAILED_STATUSES)


class CheckLimitsResult:
    CAMPAIGNS_PER_DAY_LIMIT_REACHED = 1
    EMAILS_PER_MONTH_LIMIT_REACHED = 2


def get_emails_sent_count_for_month(org_id):
    from_date = date.today() - DAYS_IN_MONTH + timedelta(days=1)
    org_stats = DeliveryErrorStats.objects.filter(
        stat_date__gte=from_date, campaign__account__org_id=org_id
    )
    sums = org_stats.aggregate(
        total_emails=Coalesce(
            Sum(Case(When(QUERY_UPLOADED_EMAILS, then="count"), default=Value(0))), 0
        ),
        failed_emails=Coalesce(
            Sum(Case(When(QUERY_FAILED_EMAILS, then="count"), default=Value(0))), 0
        ),
    )
    return sums["total_emails"] - sums["failed_emails"]


def get_emails_scheduled_count(org_id):
    sending_campaigns = Campaign.objects.filter(
        account__org_id=org_id, state=Campaign.STATUS_SENDING
    )
    return sending_campaigns.aggregate(
        scheduled_emails=Coalesce(Sum("single_use_maillist__subscribers_number"), 0)
    )["scheduled_emails"]


def get_campaigns_scheduled_count(org_id):
    return Campaign.objects.filter(account__org_id=org_id, state=Campaign.STATUS_SENDING).count()


def get_campaigns_sent_count_for_period(org_id, days):
    from_date = date.today() - timedelta(days=days - 1)
    return (
        DeliveryErrorStats.objects.filter(
            stat_date__gte=from_date, campaign__account__org_id=org_id
        )
        .distinct("campaign")
        .count()
    )


def check_campaign_fits_to_send_limits(campaign):
    exceeded, diagnostic_info = is_campaigns_per_day_send_limit_will_be_exceeded(campaign)
    if exceeded:
        return CheckLimitsResult.CAMPAIGNS_PER_DAY_LIMIT_REACHED, diagnostic_info
    reached, diagnostic_info = is_emails_per_month_send_limit_reached(campaign)
    if reached:
        return CheckLimitsResult.EMAILS_PER_MONTH_LIMIT_REACHED, diagnostic_info
    return None, {}


def is_campaigns_per_day_send_limit_will_be_exceeded(campaign):
    org_id = campaign.account.org_id
    limit_per_day = campaigns_per_day_limit(org_id)
    scheduled_count = get_campaigns_scheduled_count(org_id)
    sent_count = get_campaigns_sent_count_for_period(org_id, days=1)
    current_campaign = 1
    return scheduled_count + sent_count + current_campaign > limit_per_day, {
        "limit": "campaigns_per_day_send_limit",
        "campaigns_limit": limit_per_day,
        "campaigns_scheduled": scheduled_count,
        "campaigns_sent": sent_count,
    }


def is_emails_per_month_send_limit_reached(campaign):
    org_id = campaign.account.org_id
    limit = get_send_emails_limit(org_id)
    scheduled_count = get_emails_scheduled_count(org_id)
    sent_count = get_emails_sent_count_for_month(org_id)
    current_campaign_recipients_count = campaign.estimated_subscribers_number()
    return scheduled_count + sent_count + current_campaign_recipients_count > limit, {
        "limit": "emails_per_month_send_limit",
        "emails_limit": limit,
        "emails_scheduled": scheduled_count,
        "emails_sent": sent_count,
        "campaign_recipients": current_campaign_recipients_count,
    }


def get_test_send_task_count_for_period(org_id, days):
    from_datetime = timezone.now() - timedelta(days=days)
    campaign_ids = [
        str(id)
        for id in Campaign.objects.filter(account__org_id=org_id).values_list("id", flat=True)
    ]
    return EditorialLog.objects.filter(
        object_type="campaign",
        object_id__in=campaign_ids,
        component="test_send_task",
        action="schedule",
        datetime__gte=from_datetime,
    ).count()


def is_test_send_limit_reached(org_id):
    limit = test_sends_per_day_limit(org_id)
    return get_test_send_task_count_for_period(org_id, days=1) >= limit


def get_send_emails_limit(org_id):
    limit = _get_send_emails_limit_from_model(org_id)
    override = send_emails_limit(org_id)
    if override is not None:
        limit = override
    return limit


@atomic_auto_retry
def set_send_emails_limit(org_id, value):
    if value < 0:
        raise Exception("send emails limit is negative")
    OrganizationSettings.objects.get_or_create(org_id=org_id)
    OrganizationSettings.objects.filter(org_id=org_id).update(send_emails_limit=value)


def default_send_emails_limit():
    return settings.DEFAULT_SEND_EMAILS_LIMIT


def get_trusty(org_id):
    value = _get_trusty_from_model(org_id)
    override = trusty(org_id)
    if override is not None:
        value = override
    return value


@atomic_auto_retry
def set_trusty(org_id, value):
    if not isinstance(value, bool):
        raise Exception("value should be bool")
    OrganizationSettings.objects.get_or_create(org_id=org_id)
    OrganizationSettings.objects.filter(org_id=org_id).update(trusty=value)


def get_upgradable(org_id):
    return not get_trusty(org_id)


def is_draft_campaigns_limit_reached(org_id):
    limit = draft_campaigns_limit(org_id)
    count = get_draft_campaigns_count(org_id)
    return count >= limit


def get_maillists_count(org_id):
    return Maillist.objects.filter(account__org_id=org_id).count()


def is_maillists_limit_reached(org_id):
    limit = maillists_limit(org_id)
    count = get_maillists_count(org_id)
    return count >= limit


def _get_send_emails_limit_from_model(org_id):
    try:
        limit = OrganizationSettings.objects.get(org_id=org_id).send_emails_limit
    except OrganizationSettings.DoesNotExist:
        limit = None
    if limit is None:
        limit = default_send_emails_limit()
    return limit


def _get_trusty_from_model(org_id):
    try:
        trusty = OrganizationSettings.objects.get(org_id=org_id).trusty
    except OrganizationSettings.DoesNotExist:
        trusty = None
    if trusty is None:
        trusty = settings.DEFAULT_TRUSTY
    return trusty
