import urllib.request, urllib.parse, urllib.error
from rest_framework import status

from django.conf import settings
from django.utils.timezone import now as utcnow
from fan.utils.tvm import TvmSession
from fan.models import UserRole


def is_admin(user_id, org_id):
    resp = _do_user_request(user_id, org_id)
    if resp.status_code == status.HTTP_404_NOT_FOUND:
        return False
    resp.raise_for_status()
    resp_json = _parse_directory_response(resp)
    _validate_user_json(resp_json)
    return resp_json["is_admin"]


def is_member(user_id, org_id):
    resp = _do_user_request(user_id, org_id)
    if resp.status_code == status.HTTP_404_NOT_FOUND:
        return False
    resp.raise_for_status()
    resp_json = _parse_directory_response(resp)
    _validate_user_json(resp_json)
    return True


def get_members_among(org_id, user_ids):
    if len(user_ids) == 0:
        return []  # Directory returns all users if id param is empty
    resp = _do_users_request(org_id, user_ids)
    resp.raise_for_status()
    resp_json = _parse_directory_response(resp)
    _validate_users_response(resp_json)
    return _extract_user_ids(resp_json)


def user_has_role_in_org_accounts(user_id, org_id):
    if _does_up_to_date_role_exist(user_id, org_id):
        return True

    if is_member(user_id, org_id):
        roles_count = _user_roles_in_org_accounts(user_id, org_id).update(checked_at=utcnow())
        return bool(roles_count)
    else:
        _user_roles_in_org_accounts(user_id, org_id).delete()
    return False


def _does_up_to_date_role_exist(user_id, org_id):
    valid_after = utcnow() - settings.RECHECK_USER_ROLE_MIN_PERIOD
    return _user_roles_in_org_accounts(user_id, org_id).filter(checked_at__gt=valid_after).exists()


def _user_roles_in_org_accounts(user_id, org_id):
    return UserRole.objects.filter(account__org_id=org_id, user_id=user_id)


def _do_user_request(user_id, org_id):
    url = "{}/v11/users/{}/?fields=is_admin".format(settings.DIRECTORY_HOST, user_id)
    headers = {"X-ORG-ID": str(org_id)}
    return _do_request(url, headers)


def _parse_directory_response(resp):
    try:
        return resp.json()
    except Exception as e:
        raise Exception("Directory response is not a valid json: {}".format(e))


def _validate_user_json(resp_json):
    if "is_admin" not in resp_json:
        raise Exception('no "is_admin" field in Directory response')
    if not isinstance(resp_json["is_admin"], bool):
        raise Exception('"is_admin" field in Directory response is not bool')


def _do_users_request(org_id, user_ids):
    url = "{}/v11/users/?".format(settings.DIRECTORY_HOST) + urllib.parse.urlencode(
        {
            "id": ",".join(user_ids),
            "per_page": settings.DIRECTORY_PAGINATION_SIZE,
        }
    )
    headers = {"X-ORG-ID": str(org_id)}
    return _do_request(url, headers)


def _do_request(url, headers):
    session = TvmSession(settings.DIRECTORY_TVM_ID)
    return session.get(url, headers=headers, timeout=settings.DIRECTORY_TIMEOUT)


def _validate_users_response(resp_json):
    if "result" not in resp_json:
        raise Exception('no "result" field in Directory response')
    if not isinstance(resp_json["result"], list):
        raise Exception('"result" field in Directory response is not list')
    for user_json in resp_json["result"]:
        if "id" not in user_json:
            raise Exception('no "id" field for user in Directory response')
        if not isinstance(user_json["id"], int) and not isinstance(user_json["id"], str):
            raise Exception('"id" field of user in Directory response is not int and not unicode')


def _extract_user_ids(resp_json):
    user_ids = []
    for user_json in resp_json["result"]:
        user_ids.append(str(user_json["id"]))
    return user_ids
