from django.conf import settings
from fan.db.decorator import atomic_auto_retry
from fan.utils.emails import is_valid_localpart
from fan.models.user import UserRole


class AccountFromLoginsEmpty(Exception):
    pass


class AccountFromLoginsTooLong(Exception):
    pass


class AccountFromLoginInvalid(Exception):
    pass


class AccountUsersLimitExceeded(Exception):
    pass


@atomic_auto_retry
def set_account_from_logins(account, from_logins=settings.ACCOUNT_FROM_LOGINS_DEFAULT):
    _check_from_logins(from_logins)
    account.from_logins = _remove_duplicates(from_logins)
    account.save()


@atomic_auto_retry
def grant_account_access(account, user_ids):
    _check_account_users_limit(account, user_ids)
    for user_id in user_ids:
        UserRole.objects.get_or_create(user_id=user_id, account=account, role=UserRole.ROLES.USER)


def revoke_account_access(account, user_ids):
    UserRole.objects.filter(user_id__in=user_ids, account=account).delete()


def _check_from_logins(from_logins):
    _check_from_logins_count(from_logins)
    for from_login in from_logins:
        _check_from_login(from_login)


def _remove_duplicates(from_logins):
    return list(set(from_logins))


def _check_from_logins_count(from_logins):
    if len(from_logins) < settings.ACCOUNT_FROM_LOGINS_COUNT_MIN:
        raise AccountFromLoginsEmpty()
    if len(from_logins) > settings.ACCOUNT_FROM_LOGINS_COUNT_MAX:
        raise AccountFromLoginsTooLong()


def _check_from_login(from_login):
    if not is_valid_localpart(from_login):
        raise AccountFromLoginInvalid()


def _check_account_users_limit(account, user_ids_to_grant):
    account_users = UserRole.objects.filter(account=account, role=UserRole.ROLES.USER).values_list(
        "user_id", flat=True
    )
    users_union = set(account_users).union(set(user_ids_to_grant))
    if len(users_union) > settings.ACCOUNT_USERS_LIMIT:
        raise AccountUsersLimitExceeded()
