from django.contrib import admin
from .models import (
    OrganizationSettings,
    Account,
    AccountUnsubscribeList,
    Campaign,
    DeliveryErrorStats,
    EditorialLog,
    Letter,
    LetterAttachment,
    LetterUtm,
    Project,
    Maillist,
    SingleUseMailList,
    StatsCampaign,
    TestSendTask,
    UnsubscribeListElement,
    UserRole,
)

# Смотри документацию к админке: http://django.readthedocs.org/en/latest/ref/contrib/admin/


@admin.register(OrganizationSettings)
class OrganizationSettingsAdmin(admin.ModelAdmin):
    list_display = (
        "id",
        "org_id",
        "send_emails_limit",
        "trusty",
    )
    list_filter = (
        "send_emails_limit",
        "trusty",
    )
    search_fields = (
        "id",
        "org_id",
        "send_emails_limit",
        "trusty",
    )


@admin.register(Account)
class AccountAdmin(admin.ModelAdmin):
    list_display = ("name", "title", "status_store_period", "created_at", "modified_at")
    readonly_fields = ("created_at", "modified_at")
    search_fields = ("name",)


class CampaignLetterInline(admin.StackedInline):
    model = Letter
    extra = 0
    fields = (
        "subject",
        "from_name",
        "from_email",
        "original_letter_content_hash",
        "html_body",
    )
    can_delete = False

    def has_add_permission(self, request):
        return False


class CampaignSingleUseMailListInline(admin.StackedInline):
    model = SingleUseMailList
    extra = 0
    fields = ("description", "subscribers_number", "original_content_hash")
    readonly_fields = ("description", "subscribers_number", "original_content_hash")
    can_delete = False

    def has_add_permission(self, request):
        return False


class CampaignDeliveryErrorStatsInline(admin.TabularInline):
    model = DeliveryErrorStats
    extra = 0
    fields = ("status", "stat_date", "count")
    can_delete = False

    def has_add_permission(self, request):
        return False


class CampaignStatsInline(admin.TabularInline):
    model = StatsCampaign
    extra = 0
    fields = ("reads", "unsubscribes")
    can_delete = False

    def has_add_permission(self, request):
        return False


@admin.register(Campaign)
class CampaignAdmin(admin.ModelAdmin):
    fields = (
        "slug",
        "state",
        "maillist",
        "created_at",
        "modified_at",
    )
    readonly_fields = ("created_at", "modified_at")
    save_on_top = True
    list_display = (
        "id",
        "org_id",
        "slug",
        "from_email",
        "from_name",
        "subject",
        "state",
        "created_at",
        "modified_at",
    )
    list_filter = ("state",)
    inlines = [
        CampaignLetterInline,
        CampaignSingleUseMailListInline,
        CampaignDeliveryErrorStatsInline,
        CampaignStatsInline,
    ]
    search_fields = (
        "id",
        "account__org_id",
        "slug",
        "letters_rel__from_email",
        "state",
    )

    def org_id(self, instance):
        return instance.account.org_id

    org_id.admin_order_field = "account__org_id"


@admin.register(Letter)
class LetterAdmin(admin.ModelAdmin):
    list_display = (
        "code",
        "subject",
        "from_name",
        "from_email",
        "reply_to",
        "created_at",
        "modified_at",
    )
    search_fields = ("id", "subject")
    readonly_fields = ("created_at", "modified_at")


@admin.register(LetterAttachment)
class LetterAttachmentAdmin(admin.ModelAdmin):
    list_display = ("letter", "uri", "mime_type", "subtype", "created_at", "modified_at")
    search_fields = ("letter_id",)
    readonly_fields = ("created_at", "modified_at")


@admin.register(LetterUtm)
class LetterUtmAdmin(admin.ModelAdmin):
    list_display = ("letter", "utm_source", "utm_medium", "utm_campaign", "utm_content")
    search_fields = ("letter_id",)


@admin.register(EditorialLog)
class EditorialLogAdmin(admin.ModelAdmin):
    list_display = (
        "object_type",
        "object_id",
        "component",
        "action",
        "description",
        "metadata",
        "user",
        "user_ip",
    )
    search_fields = ("object_id",)


@admin.register(SingleUseMailList)
class SingleUseMailListAdmin(admin.ModelAdmin):
    list_display = ("id", "campaign", "subscribers_number", "created_at", "modified_at")
    search_fields = ("campaign_id",)
    readonly_fields = ("created_at", "modified_at")


class CampaignInline(admin.TabularInline):
    model = Campaign
    extra = 0
    fields = (
        "id",
        "slug",
        "from_email",
        "from_name",
        "subject",
        "state",
        "created_at",
        "modified_at",
    )
    readonly_fields = (
        "id",
        "slug",
        "from_email",
        "from_name",
        "subject",
        "state",
        "created_at",
        "modified_at",
    )
    can_delete = False

    def from_email(self, instance):
        return instance.from_email

    def from_name(self, instance):
        return instance.from_name

    def subject(self, instance):
        return instance.subject

    def has_add_permission(self, request):
        return False


@admin.register(Maillist)
class MaillistAdmin(admin.ModelAdmin):
    search_fields = ("id", "account__org_id", "slug", "title", "filename", "size")
    list_display = (
        "id",
        "org_id",
        "slug",
        "title",
        "filename",
        "size",
        "created_at",
        "modified_at",
    )
    list_filter = ("account__org_id",)
    readonly_fields = ("created_at", "modified_at")
    inlines = [CampaignInline]

    def org_id(self, instance):
        return instance.account.org_id

    org_id.admin_order_field = "account__org_id"


for model in (Project,):
    admin.site.register(model)


@admin.register(UserRole)
class UserRoleAdmin(admin.ModelAdmin):
    search_fields = ("user__username", "user__email", "account__name")
    list_display = ("id", "account", "user_id", "role", "created_at", "modified_at")
    readonly_fields = ("created_at", "modified_at")


@admin.register(DeliveryErrorStats)
class DeliveryErrorStatsAdmin(admin.ModelAdmin):
    ordering = ["-campaign_id"]
    search_fields = ("campaign__id", "letter__id")
    list_display = ("id", "letter", "status", "count")


@admin.register(AccountUnsubscribeList)
class AccountUnsubscribeListAdmin(admin.ModelAdmin):
    list_display = ("get_name_locale", "account", "visible")
    list_filter = ("account",)
    list_editable = ("visible",)

    def get_name_locale(self, instance):
        return instance.get_name()

    get_name_locale.short_description = "Название"


@admin.register(UnsubscribeListElement)
class UnsubscribeListElementAdmin(admin.ModelAdmin):
    list_display = ("email", "list", "created_at", "modified_at")
    list_filter = ("list",)
    search_fields = ("email",)
    readonly_fields = ("created_at", "modified_at")


@admin.register(TestSendTask)
class TestSendTaskAdmin(admin.ModelAdmin):
    pass


@admin.register(StatsCampaign)
class StatsCampaignAdmin(admin.ModelAdmin):
    pass
