from zlib import decompress
from django.core.exceptions import MultipleObjectsReturned
from fan.models import Campaign, SingleUseMailList


def get_campaign_by_slug(slug, account_id):
    return Campaign.objects.get(slug=slug, account_id=account_id)


def get_locked_campaign_by_slug(slug, account_id, using="default"):
    query_result = (
        Campaign.objects.using(using)
        .filter(slug=slug, account_id=account_id)
        .select_for_update(nowait=True)
    )
    if len(query_result) == 0:
        raise Campaign.DoesNotExist()
    if len(query_result) > 1:
        raise MultipleObjectsReturned()
    return query_result[0]


def get_draft_campaigns_count(org_id):
    return Campaign.objects.filter(account__org_id=org_id, state=Campaign.STATUS_DRAFT).count()


def get_csv_data(campaign):
    if campaign.maillist:
        return _get_csv_data_from_maillist(campaign)
    return _get_csv_data_from_singleusemaillist(campaign)


def _get_csv_data_from_maillist(campaign):
    return decompress(campaign.maillist.data).decode()


def _get_csv_data_from_singleusemaillist(campaign):
    try:
        return decompress(campaign.single_use_maillist.data).decode()
    except SingleUseMailList.DoesNotExist:
        return None
