from datetime import date
from django.conf import settings
from fan.accounts.get import get_account_from_logins, need_to_silently_drop_sents
from fan.accounts.organizations.domains import CheckDomainResult, check_campaign_domain
from fan.campaigns.exceptions import (
    DomainNotBelongs,
    DomainNoMX,
    DomainNoDKIM,
    DomainNoSPF,
    EmptyFromEmail,
    EmptyFromName,
    EmptySubject,
    ForbiddenCurrentCampaignState,
    ForbiddenTargetCampaignState,
    InvalidStatValue,
    LoginNotBelongs,
    MismatchedStatsSummary,
    NoLetter,
    NoMaillist,
    UnsupportedStat,
)
from fan.campaigns.get import get_locked_campaign_by_slug
from fan.db.decorator import atomic_auto_retry
from fan.delivery.status import DeliveryStatus
from fan.models import Campaign, DeliveryErrorStats
from fan.utils.emails import get_local_part


SEND_STAT_KEYS = {
    "email_uploaded": DeliveryStatus.EMAIL_UPLOADED,
    "email_unsubscribed": DeliveryStatus.EMAIL_UNSUBSCRIBED,
    "email_duplicated": DeliveryStatus.EMAIL_DUPLICATED,
    "email_invalid": DeliveryStatus.EMAIL_INVALID,
    "undefined_error": DeliveryStatus.UNDEFINED_ERROR,
}


@atomic_auto_retry
def set_campaign_details(campaign, campaign_details):
    _check_allowed_src_campaign_state_param(campaign)
    letter = campaign.default_letter
    if "from_email" in campaign_details:
        letter.from_email = campaign_details["from_email"]
    if "from_name" in campaign_details:
        letter.from_name = campaign_details["from_name"]
    if "subject" in campaign_details:
        letter.subject = campaign_details["subject"]
    letter.save()


def clone_campaign_details(campaign, source_campaign):
    source_letter = source_campaign.default_letter
    set_campaign_details(
        campaign,
        {
            "from_email": source_letter.from_email,
            "from_name": source_letter.from_name,
            "subject": source_letter.subject,
        },
    )


@atomic_auto_retry
def set_campaign_state(campaign, state):
    _check_allowed_dst_campaign_state_param(state)
    _check_allowed_src_campaign_state_param(campaign)
    _check_campaign_letter(campaign)
    _check_campaign_maillist(campaign)
    if settings.CHECK_CAMPAIGN_FROM_LOGIN:
        _check_campaign_from_login(campaign)
    _check_campaign_domain(campaign)
    if need_to_silently_drop_sents(campaign.account):
        return _silently_drop_campaign(campaign)
    campaign.state = state
    campaign.save()


def set_campaign_send_result(campaign, state, result):
    if state == Campaign.STATUS_SENT:
        _check_campaign_sent_stats(result)
        _set_campaign_sent(campaign, result)
    elif state == Campaign.STATUS_FAILED:
        _set_campaign_failed(campaign)
    else:
        raise ForbiddenTargetCampaignState()


def _check_campaign_sent_stats(stats):
    _check_unsupported_sent_stats(stats)
    _check_invalid_sent_stats(stats)
    _check_sent_stats_summary(stats)


def _check_unsupported_sent_stats(stats):
    for key, value in list(stats.items()):
        if key not in list(SEND_STAT_KEYS.keys()):
            raise UnsupportedStat(key)


def _check_invalid_sent_stats(stats):
    for stat, value in list(stats.items()):
        if not isinstance(value, int):
            raise InvalidStatValue(stat)


def _check_sent_stats_summary(stats):
    size = stats.get("email_uploaded", 0)
    summary = 0
    for stat, value in list(stats.items()):
        if stat != "email_uploaded":
            summary += value
    if size < summary:
        raise MismatchedStatsSummary()


@atomic_auto_retry
def _set_campaign_sent(campaign, stats):
    campaign = get_locked_campaign_by_slug(campaign.slug, campaign.account.id)
    _check_campaign_is_sending(campaign)
    _set_campaign_sent_stats(campaign, stats)
    campaign.state = Campaign.STATUS_SENT
    campaign.save()


@atomic_auto_retry
def _set_campaign_failed(campaign):
    campaign = get_locked_campaign_by_slug(campaign.slug, campaign.account.id)
    _check_campaign_is_sending(campaign)
    campaign.state = Campaign.STATUS_FAILED
    campaign.save()


def _check_campaign_is_sending(campaign):
    if campaign.state != Campaign.STATUS_SENDING:
        raise ForbiddenCurrentCampaignState(campaign.state, Campaign.STATUS_SENDING)


def _set_campaign_sent_stats(campaign, stats):
    stat_date = date.today()
    for key, value in list(SEND_STAT_KEYS.items()):
        count = stats.get(key, 0)
        _set_campaign_sent_stat(stat_date, campaign, value, count)


def _set_campaign_sent_stat(date, campaign, status, count):
    stat, _ = DeliveryErrorStats.objects.get_or_create(
        stat_date=date, campaign=campaign, letter=campaign.default_letter, status=status
    )
    stat.count = count
    stat.save()


def _check_allowed_dst_campaign_state_param(state):
    if state != Campaign.STATUS_SENDING:
        raise ForbiddenTargetCampaignState()


def _check_allowed_src_campaign_state_param(campaign):
    if campaign.state != Campaign.STATUS_DRAFT:
        raise ForbiddenCurrentCampaignState(campaign.state, Campaign.STATUS_DRAFT)


def _check_campaign_letter(campaign):
    if not campaign.letter_uploaded:
        raise NoLetter()
    if not campaign.from_email:
        raise EmptyFromEmail()
    if not campaign.from_name:
        raise EmptyFromName()
    if not campaign.subject:
        raise EmptySubject()


def _check_campaign_domain(campaign):
    res = check_campaign_domain(campaign)
    if res == CheckDomainResult.NOT_BELONGS:
        raise DomainNotBelongs()
    if res == CheckDomainResult.NO_MX:
        raise DomainNoMX()
    if res == CheckDomainResult.NO_DKIM:
        raise DomainNoDKIM()
    if res == CheckDomainResult.NO_SPF:
        raise DomainNoSPF()


def _check_campaign_from_login(campaign):
    from_logins = get_account_from_logins(campaign.account)
    login = get_local_part(campaign.from_email)
    if login not in from_logins:
        raise LoginNotBelongs()


def _check_campaign_maillist(campaign):
    if not campaign.maillist_uploaded:
        raise NoMaillist()


def _silently_drop_campaign(campaign):
    _set_campaign_sent_stats(campaign, {"email_uploaded": campaign.estimated_subscribers_number()})
    campaign.state = Campaign.STATUS_SENT
    campaign.save()
