from functools import wraps
from django.db import transaction
from .postgres.helpers import CONNECTION_ERRORS, is_connection_exception


def atomic_auto_retry(func):
    """
    Данные декоратор обернет функцию в транзакцию.
    При этом, если в процессе возникнет ошибка соединения: переоткроет соединение и сделает 1 ретрай функции
    """

    @wraps(func)
    def func_wrapper(*args, **kwargs):
        transact_func = transaction.atomic(func)

        try:
            return transact_func(*args, **kwargs)
        except CONNECTION_ERRORS as exc:
            connection = transaction.get_connection()

            if not connection.in_atomic_block and is_connection_exception(exc):
                # Делаем ретрай, только в том случае, когда мы целиком вышли из atomic блока
                if hasattr(connection, "close"):
                    connection.close(reconnect=True)

                return transact_func(*args, **kwargs)
            else:
                raise

    return func_wrapper
