from django.conf import settings
from django.db.models import Field
from django.db.models.lookups import BuiltinLookup
from django.db.models.query import EmptyResultSet


class ILike(BuiltinLookup):
    """
    In SQL it looks like:
        ... "fan_account"."name" COLLATE "en_US.utf8" ILIKE %a% COLLATE "en_US.utf8" ...
    """

    lookup_name = "ilike"  # case insensitive like
    collation = settings.ILIKE_COLLATION

    def as_sql(self, compiler, connection):
        lhs_sql, params = self.process_lhs(compiler, connection)
        rhs_sql, rhs_params = self.process_rhs(compiler, connection)
        params.extend(rhs_params)
        rhs_sql = self.get_rhs_op(connection, rhs_sql)

        # '{lhs} COLLATE "{collation}" {rhs}'
        # where rhs = 'ILIKE %{param[0]}% COLLATE "{collation}" ...'
        return '%s COLLATE "%s" %s' % (lhs_sql, self.collation, rhs_sql), params

    # see Contains lookup definition
    def process_rhs(self, qn, connection):
        rhs, params = super().process_rhs(qn, connection)
        if params and not self.bilateral_transforms:
            params[0] = "%%%s%%" % connection.ops.prep_for_like_query(params[0])
        return rhs, params

    def get_rhs_op(self, connection, rhs):
        return 'ILIKE %s COLLATE "%s"' % (rhs, self.collation)


class UpperIn(BuiltinLookup):
    """
    Позволяет фильтровать записи по одному из нескольких строковых значений сравнением через UPPER
    Полезно в ситуации, когда на таблице есть индекс, включающий нужную строковую колонку, например:
        UNIQUE, btree (list_id, upper(email::text))

    field__upperin=['a', 'b'] -> UPPER(field::text) IN (UPPER('a'::text), UPPER('b'::text))

    Например
    >>> UnsubscribeListElement.objects.filter(
    >>>     email__upperin=['alice@example.com', 'bob@example.com'], list_id__in=[1, 2]
    >>> ).values('email')
    Породит запрос
    SELECT "fan_unsubscribelistelement"."email"
    FROM "fan_unsubscribelistelement"
    WHERE (
        UPPER("fan_unsubscribelistelement"."email"::text) IN (
            UPPER('alice@example.com'::text), UPPER('bob@example.com'::text)
        )
        AND "fan_unsubscribelistelement"."list_id" IN (1, 2)
    )
    """

    lookup_name = "upperin"

    def as_sql(self, compiler, connection):
        lhs_sql, params = self.process_lhs(compiler, connection)
        rhs_sql, rhs_params = self.process_rhs(compiler, connection)
        params.extend(rhs_params)
        rhs_sql = self.get_rhs_op(connection, rhs_sql)
        return "UPPER(%s::text) %s" % (lhs_sql, rhs_sql), params

    def get_rhs_op(self, connection, rhs):
        return "IN %s" % rhs

    def process_rhs(self, qn, connection):
        if self.rhs_is_direct_value():
            sql_params = list(self.rhs)

            if not sql_params:
                raise EmptyResultSet

            sql = "(%s)" % ", ".join("UPPER(%s::text)" for _ in sql_params)
            return sql, sql_params
        else:
            return super().process_rhs(qn, connection)

    def get_prep_lookup(self):
        return self.rhs


Field.register_lookup(ILike)
Field.register_lookup(UpperIn)
