"""
pythoncompat
"""


import sys
import collections

# -------
# Pythons
# -------

# Syntax sugar.
_ver = sys.version_info

#: Python 2.x?
is_py2 = _ver[0] == 2

#: Python 3.x?
is_py3 = _ver[0] == 3


# ---------
# Specifics
# ---------

if is_py2:

    chr = chr
    text_type = str
    string_types = (str, str)
    integer_types = (int, int)
    int_to_byte = chr

    import urllib.parse
    from .ordereddict import OrderedDict

    from io import StringIO
    from io import StringIO as BytesIO

    NativeStringIO = BytesIO

    def to_native(x, charset=sys.getdefaultencoding(), errors="strict"):
        if x is None or isinstance(x, str):
            return x
        return x.encode(charset, errors)

    def is_callable(x):
        return isinstance(x, collections.Callable)

    def to_bytes(x, charset=sys.getdefaultencoding(), errors="strict"):
        if x is None:
            return None
        if isinstance(x, (bytes, bytearray, buffer)):
            return bytes(x)
        if isinstance(x, str):
            return x.encode(charset, errors)
        raise TypeError("Expected bytes")

    from email.utils import formataddr


elif is_py3:
    import urllib.parse as urlparse
    from collections import OrderedDict, Callable
    from io import StringIO, BytesIO

    NativeStringIO = StringIO

    chr = chr
    text_type = str
    string_types = (str,)
    integer_types = (int,)

    def to_native(x, charset=sys.getdefaultencoding(), errors="strict"):
        if x is None or isinstance(x, str):
            return x
        return x.decode(charset, errors)

    def is_callable(x):
        return isinstance(x, Callable)

    def to_bytes(x, charset=sys.getdefaultencoding(), errors="strict"):
        if x is None:
            return None
        if isinstance(x, (bytes, bytearray, memoryview)):
            return bytes(x)
        if isinstance(x, str):
            return x.encode(charset, errors)
        raise TypeError("Expected bytes")

    from email.utils import escapesre, specialsre

    def formataddr(pair):
        """
        This code is copy of python2 email.utils.formataddr.
        Takes a 2-tuple of the form (realname, email_address) and returns RFC2822-like string.
        Does not encode non-ascii realname.

        Python3 email.utils.formataddr do encode realname.
        """
        name, address = pair
        if name:
            quotes = ""
            if specialsre.search(name):
                quotes = '"'
            name = escapesre.sub(r"\\\g<0>", name)
            return "%s%s%s <%s>" % (quotes, name, quotes, address)
        return address


def to_unicode(x, charset=sys.getdefaultencoding(), errors="strict", allow_none_charset=False):
    if x is None:
        return None
    if not isinstance(x, bytes):
        return text_type(x)
    if charset is None and allow_none_charset:
        return x
    return x.decode(charset, errors)
