import os.path

from ..message import Message
from .local_store import ZipLoader, FileNotFound


class LoadError(Exception):
    pass


class IndexFileNotFound(LoadError):
    pass


class InvalidHtmlFile(LoadError):
    pass


def from_html(
    html,
    text=None,
    base_url=None,
    message_params=None,
    local_loader=None,
    template_cls=None,
    message_cls=None,
    source_filename=None,
    requests_params=None,
    **kwargs
):

    """
    Loads message from html string with images from local_loader.

    :param html: html string
    :param base_url: base_url for html
    :param text: text string or None
    :param template_cls: if set, html and text are set with this template class
    :param local_loader: loader with local files
    :param message_cls: default is emails.Message
    :param message_params: parameters for Message constructor
    :param source_filename: source html file name (used for exception description on html parsing error)
    :param requests_params: parameters for external url handling
    :param kwargs: arguments for transformer.load_and_transform
    :return:
    """

    if template_cls is None:
        template_cls = lambda x: x

    message_params = message_params or {}

    _param_html = message_params.pop("html", None)
    _param_text = message_params.pop("text", None)

    message = (message_cls or Message)(
        html=template_cls(html or _param_html or ""),
        text=template_cls(text or _param_text),
        **message_params
    )
    message.create_transformer(
        requests_params=requests_params, base_url=base_url, local_loader=local_loader
    )
    if message.transformer.tree is None:
        raise InvalidHtmlFile("Error parsing '%s'" % source_filename)
    message.transformer.load_and_transform(**kwargs)
    message.transformer.save()
    message._loader = local_loader
    return message


def _from_filebased_source(
    store,
    skip_html=False,
    html_filename=None,
    skip_text=True,
    text_filename=None,
    message_params=None,
    **kwargs
):
    """
    Loads message from prepared store `store`.

    :param store: prepared filestore
    :param skip_html: if True, make message without html part
    :param html_filename: html part filename. If None, search automatically.
    :param skip_text: if True, make message without text part
    :param text_filename: text part filename. If None, search automatically.
    :param message_params: parameters for Message
    :param kwargs: arguments for from_html
    :return:
    """

    if not skip_html:
        try:
            html_filename = store.find_index_html(html_filename)
        except FileNotFound:
            raise IndexFileNotFound("html file not found")

    dirname, html_filename = os.path.split(html_filename)
    if dirname:
        store.base_path = dirname

    html = store.content(html_filename, is_html=True, guess_charset=True)

    text = None
    if not skip_text:
        text_filename = store.find_index_text(text_filename)
        text = text_filename and store.content(text_filename) or None

    return from_html(
        html=html,
        text=text,
        local_loader=store,
        source_filename=html_filename,
        message_params=message_params,
        **kwargs
    )


def from_zip(zip_file, loader_cls=None, **kwargs):
    """
    Loads message from zipfile.

    :param zip_file: file-like object with zip file
    :param kwargs: arguments for _from_filebased_source function
    :return: emails.Message object
    """
    loader_cls = loader_cls or ZipLoader
    return _from_filebased_source(store=loader_cls(file=zip_file), **kwargs)
