import os
from os import path
from zipfile import ZipFile

from ..compat import to_unicode

from ..loader.helpers import decode_text


class FileNotFound(Exception):
    pass


def split_template_path(template):
    """Split a path into segments and perform a sanity check.  If it detects
    '..' in the path it will raise a `TemplateNotFound` error.
    """
    pieces = []
    for piece in template.split("/"):
        if path.sep in piece or (path.altsep and path.altsep in piece) or piece == path.pardir:
            raise FileNotFound(template)
        elif piece and piece != ".":
            pieces.append(piece)
    return pieces


class BaseLoader:
    def __getitem__(self, filename):
        try:
            contents, _ = self.get_file(filename)
            return contents
        except FileNotFound:
            return None

    def get_file(self, name):
        raise NotImplementedError

    def list_files(self):
        raise NotImplementedError

    def content(self, filename, is_html=False, decode=True, guess_charset=False, charset="utf-8"):
        data = self[filename]
        if decode:
            data, encoding = decode_text(
                data,
                is_html=is_html,
                guess_charset=guess_charset,
                try_common_charsets=False,
                fallback_charset=charset,
            )
        return data

    def find_index_file(
        self,
        filename=None,
        extensions=(".html", ".htm"),
        stop_names=("index",),
        raise_if_not_found=True,
    ):

        if filename:
            if self[filename]:
                return filename
            else:
                raise FileNotFound(filename)

        found_files = []

        for filename in self.list_files():

            bn = os.path.basename(filename)
            if bn.startswith("."):
                # ignore hidden files
                continue
            name, ext = os.path.splitext(bn)

            if ext in extensions:
                if stop_names and name in stop_names:
                    return filename
                found_files.append(filename)

        # Return first found file
        if found_files:
            return found_files[0]
        elif raise_if_not_found:
            raise FileNotFound("index %s" % "|".join(extensions))

    def find_index_html(self, filename=None):
        return self.find_index_file(filename=filename)

    def find_index_text(self, filename=None):
        return self.find_index_file(
            filename=filename, extensions=(".txt",), stop_names=("index",), raise_if_not_found=False
        )


class ZipLoader(BaseLoader):

    """
    Load files from zip file
    """

    common_filename_charsets = ["ascii", "cp866", "cp1251", "utf-8"]

    def __init__(self, file, encoding="utf-8", base_path=None):

        if not isinstance(file, ZipFile):
            file = ZipFile(file, "r")

        self.zipfile = file
        self.encoding = encoding
        self.base_path = base_path
        self._decoded_filenames = None
        self._original_filenames = None

    def _decode_filename(self, name):
        for enc in self.common_filename_charsets:
            try:
                return to_unicode(name, enc)
            except UnicodeDecodeError:
                pass
        return name

    def _unpack(self):
        if self._decoded_filenames is None:
            self._original_filenames = set(self.zipfile.namelist())
            self._decoded_filenames = dict(
                [(self._decode_filename(name), name) for name in self._original_filenames]
            )

    def get_file(self, name):

        if self.base_path:
            name = path.join(self.base_path, name)

        name = path.join(*split_template_path(name))

        self._unpack()

        if isinstance(name, str):
            name = to_unicode(name, "utf-8")

        if name not in self._original_filenames:
            name = self._decoded_filenames.get(name)

        if name is None:
            raise FileNotFound(name)

        return self.zipfile.read(name), name

    def list_files(self):
        self._unpack()
        return sorted(self._decoded_filenames)
