import logging
import uuid
from io import BytesIO

from django.conf import settings

import requests

logger = logging.getLogger(__name__)

DEFAULT_CONNECTION_TIMEOUT = 5


class AvatarsPublisher:
    """
    Класс для публикации файлов в аватарнице
    """

    _PUT_URL_TEMPLATE = "http://{host}/put-{namespace}/{name}"
    _DELETE_URL_TEMPLATE = "http://{host}/delete-{namespace}/{common_path}"

    def __init__(self):
        self._host = settings.AVATARS_HOSTS["write"]
        self._namespace = settings.AVATARS_NAMESPACE
        self.timeout = settings.AVATARS_TIMEOUT
        self._session = requests.session()

    @staticmethod
    def get_read_url(path, use_ssl=True):
        schema = "https" if use_ssl else "http"
        return "{schema}://{host}{path}".format(
            schema=schema, host=settings.AVATARS_HOSTS["read"], path=path
        )

    def publish(self, data):
        return self.publish_file(BytesIO(data))

    def publish_file(self, file_):
        """
        Possible avatars errors looks:
        status code = 500
        body =
        {
            "attrs": {},
            "description": "cannot upload data due to internal error",
            "status": "error"
        }
        """
        r = self._session.post(
            self._make_put_url(self._make_name()),
            files={"file": file_},
            timeout=self.timeout,
        )
        r.raise_for_status()
        return self._get_path(r)

    def clone(self, path):
        return self.publish_url(self.get_read_url(path))

    def publish_url(self, url):
        r = self._session.get(
            self._make_put_url(self._make_name()),
            params={"url": url},
            timeout=self.timeout,
        )
        r.raise_for_status()
        return self._get_path(r)

    def unpublish(self, path):
        path_parts = path.split("/")
        if len(path_parts) < 5:
            return
        common_path = "/".join(path_parts[2:])

        url = self._DELETE_URL_TEMPLATE.format(
            host=self._host, namespace=self._namespace, common_path=common_path
        )

        try:
            self._session.get(url, timeout=self.timeout).raise_for_status()
        except Exception:
            return

    def _make_name(self):
        return str(uuid.uuid4())

    def _make_put_url(self, name):
        return self._PUT_URL_TEMPLATE.format(host=self._host, namespace=self._namespace, name=name)

    def _get_path(self, response):
        return response.json()["sizes"]["orig"]["path"]
