from zlib import compress, decompress
from django.conf import settings
from django.db import transaction
from fan.accounts.organizations.limits import is_maillists_limit_reached
from fan.campaigns.exceptions import ForbiddenCurrentCampaignState
from fan.lists.csv_maillist import get_preview, get_subscribers_number, parse_csv_data
from fan.models import Campaign, Maillist, SingleUseMailList


class MaillistsCountExceeded(Exception):
    pass


class MaillistTitleLengthExceeded(Exception):
    pass


class MaillistTitleDuplicated(Exception):
    pass


class MaillistDoesNotExist(Exception):
    pass


def get_maillist(account, slug):
    try:
        return Maillist.objects.get(account=account, slug=slug)
    except Maillist.DoesNotExist:
        raise MaillistDoesNotExist


def set_maillist_details(maillist, title):
    _validate_maillist_title(maillist.account, title=title)
    maillist = _set_maillist_title(maillist, title=title)
    return maillist


def get_maillists(account):
    return account.maillists.all()


@transaction.atomic
def store_maillist(account, csv_data, filename, campaign=None, title=None):
    _check_maillists_count_limit_reached(account)
    _validate_campaign(account, campaign=campaign)
    _validate_maillist_title(account, title=title)
    csv_maillist = parse_csv_data(csv_data)
    maillist = Maillist.objects.create(
        account=account,
        filename=filename,
        data=compress(csv_data.encode("utf-8")),
        preview=get_preview(csv_maillist),
        size=get_subscribers_number(csv_maillist),
    )
    set_maillist_for_campaign(campaign, maillist)
    maillist = _set_maillist_title(maillist, title=title)
    return maillist


def retrieve_maillist_data(maillist):
    return decompress(maillist.data).decode("utf-8")


def delete_maillist(maillist):
    maillist.delete()


@transaction.atomic
def set_maillist_for_campaign(campaign, maillist):
    if not campaign:
        return campaign
    if maillist.account != campaign.account:
        raise RuntimeError("maillist does not belong to campaign's account")
    if campaign.state != Campaign.STATUS_DRAFT:
        raise ForbiddenCurrentCampaignState(campaign.state, Campaign.STATUS_DRAFT)
    _delete_single_use_maillist(campaign)
    campaign.maillist = maillist
    campaign.save()
    campaign = Campaign.objects.get(id=campaign.id)
    return campaign


def _check_maillists_count_limit_reached(account):
    if is_maillists_limit_reached(account.org_id):
        raise MaillistsCountExceeded()


def _delete_single_use_maillist(campaign):
    try:
        campaign.single_use_maillist.delete()
    except SingleUseMailList.DoesNotExist:
        pass


def _validate_campaign(account, campaign=None):
    if not campaign:
        return
    if campaign.account != account:
        raise RuntimeError("campaign does not belong to account")


def _validate_maillist_title(account, title=None):
    if title and len(title) > settings.MAILLIST_TITLE_MAX_LENGTH:
        raise MaillistTitleLengthExceeded()
    if title and Maillist.objects.filter(account=account, title=title).exists():
        raise MaillistTitleDuplicated()


def _set_maillist_title(maillist, title=None):
    if not title:
        title = settings.MAILLIST_TITLE_DEFAULT.format(maillist_slug=maillist.slug)
    maillist.title = title
    maillist.save()
    return maillist
