from zlib import compress
from django.db import transaction
from fan.campaigns.exceptions import ForbiddenCurrentCampaignState
from fan.lists.csv_maillist import get_preview, get_subscribers_number, parse_csv_data
from fan.models import Campaign, SingleUseMailList
from fan.utils.hash import strhash


@transaction.atomic
def store_csv_maillist_for_campaign(campaign, csv_data, description):
    if campaign.state != Campaign.STATUS_DRAFT:
        raise ForbiddenCurrentCampaignState(campaign.state, Campaign.STATUS_DRAFT)
    _unset_maillist_for_campaign(campaign)
    _find_or_create_maillist(campaign, csv_data, description)


def clone_csv_for_campaign(campaign, source_campaign):
    if campaign.state != Campaign.STATUS_DRAFT:
        raise ForbiddenCurrentCampaignState(campaign.state, Campaign.STATUS_DRAFT)
    source_maillist = _get_maillist(source_campaign)
    if source_maillist:
        _clone_maillist(campaign, source_maillist)
    else:
        _delete_maillist(campaign)


def _unset_maillist_for_campaign(campaign):
    campaign.maillist = None
    campaign.save()


def _find_or_create_maillist(campaign, csv_data, description):
    csv_maillist = parse_csv_data(csv_data)
    data_fields = {
        "original_content_hash": strhash(csv_data),
        "data_encoding": "gzip",
        "data": compress(csv_data.encode("utf-8")),
        "data_preview": get_preview(csv_maillist),
        "description": description,
        "subscribers_number": get_subscribers_number(csv_maillist),
    }
    SingleUseMailList.objects.update_or_create(campaign=campaign, defaults=data_fields)


def _get_maillist(campaign):
    try:
        return campaign.single_use_maillist
    except SingleUseMailList.DoesNotExist:
        return None


def _clone_maillist(campaign, source_maillist):
    skip_fields = ["id", "campaign", "created_at", "modified_at"]
    cloned_fields = {}
    for field in SingleUseMailList._meta.get_fields():
        if field.name in skip_fields:
            continue
        cloned_fields[field.name] = getattr(source_maillist, field.name)
    SingleUseMailList.objects.update_or_create(campaign=campaign, defaults=cloned_fields)


def _delete_maillist(campaign):
    try:
        campaign.single_use_maillist.delete()
    except SingleUseMailList.DoesNotExist:
        pass
