import datetime
import logging

import jinja2.sandbox
import ujson as json
from jinja2 import Undefined
from jinja2.exceptions import FilterArgumentError
from jinja2.exceptions import TemplateSyntaxError, UndefinedError

from .exceptions import TemplateRuntimeError
from .template_extensions import registered as registered_extensions


class CustomEnvironment(jinja2.sandbox.SandboxedEnvironment):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.tests["gte"] = lambda a, b: a >= b
        self.tests["gt"] = lambda a, b: a > b
        self.tests["lte"] = lambda a, b: a <= b
        self.tests["lt"] = lambda a, b: a < b
        self.tests["eq"] = lambda a, b: a == b
        self.tests["ne"] = lambda a, b: a != b

        for ext in registered_extensions:
            self.add_extension(ext)

    def _format_exception_error(self, exc):
        if isinstance(exc, TemplateSyntaxError):
            return 'Template syntax error:"{message}" at line: {lineno}'.format(
                message=exc.message, lineno=exc.lineno
            )
        if isinstance(exc, UndefinedError):
            return 'Template improper use of UNDEFINED:"{message}"'.format(message=exc.message)
        return 'Template runtime error:"{message}"'.format(message=str(exc))

    def handle_exception(self, exc_info=None, rendered=False, source_hint=None):
        try:
            super().handle_exception(exc_info, rendered, source_hint)
        except Exception as exc:
            message = self._format_exception_error(exc)
            logging.exception(message)
            raise TemplateRuntimeError(message)


env = CustomEnvironment()


@jinja2.contextfilter
def required(context, value):
    if isinstance(value, Undefined) or (value is None):
        raise UndefinedError("Required value missed")
    return value


def json_decode(value):
    if isinstance(value, str):
        try:
            return json.loads(value)
        except (ValueError, TypeError):
            raise FilterArgumentError("Wrong JSON object")

    return value


def strftime(value, fmt):
    """Convert date / datetime object to string using given format string."""
    if not isinstance(value, datetime.date):
        raise FilterArgumentError("strftime: argument is not date or datetime object.")
    if not isinstance(fmt, str):
        raise FilterArgumentError(
            "strftime: format argument is not of string type - got {t} instead.".format(t=type(fmt))
        )
    return value.strftime(fmt)


def strptime(date_string, fmt):
    """Reconstruct datetime object form given string and format."""
    if not isinstance(date_string, str):
        raise FilterArgumentError("strptime: argument is not of string type.")
    if not isinstance(fmt, str):
        raise FilterArgumentError(
            "strptime: format argument is not of string type - got {t} instead.".format(t=type(fmt))
        )
    return datetime.datetime.strptime(date_string, fmt)


for (name, func) in (
    ("required", required),
    ("json_decode", json_decode),
    ("strftime", strftime),
    ("strptime", strptime),
):
    env.filters[name] = func
