import logging

from django.conf import settings
from django.db import transaction
from urllib.parse import urlparse

from fan.file_models import AvatarsPublisher
from fan.links.statistic_wrapper import (
    get_local_images_urls_from_html,
    get_web_images_urls_from_html,
    substitute_images_paths,
)
from fan.links.unsubscribe import search_unsubscribe_link
from fan.message.body_upload import load_message_from_file
from fan.message.exceptions import (
    UnsubscribeLinkNotFoundError,
    ModelRuntimeError,
    TemplateRuntimeError,
)
from fan.message.render import test_render
from fan.models.letter import LetterAttachment
from fan.utils.hash import filehash
from fan.utils.persistent_log import persistent_log


CONTENT_TYPE_HTML = "text/html"
CONTENT_TYPE_ZIP = "application/zip"
CONTENT_TYPE_PNG = "image/png"
CONTENT_TYPE_MULTIPART = "multipart/form-data"

SUPPORTED_CONTENT_TYPES = (
    CONTENT_TYPE_HTML,
    CONTENT_TYPE_ZIP,
)


class AttachmentsCountExceeded(Exception):
    pass


class AttachmentsSizeExceeded(Exception):
    pass


class UserTemplateVariablesCountExceeded(Exception):
    pass


class UserTemplateVariableLengthExceeded(Exception):
    pass


def load_letter(letter, html_body_file, request=None):
    message = _load_message(letter=letter, html_body_file=html_body_file)

    _validate_html_local_images_are_attached(message)
    _validate_attachments_total_count(letter, message)
    _validate_attachments_total_size(letter, message)

    _publish_new_attachments(message.attachments)
    _substitute_html_local_images_urls_with_published(message)

    @transaction.atomic
    def db_upload():
        _upload_new_attachments(letter, message.attachments)
        upload_letter_html_body(letter, message.html)
        _upload_letter_meta(letter, html_body_file)
        _drop_unused_attachments(letter.attachments.all(), message)

    try:
        db_upload()
    except:
        _unpublish_new_attachments(message.attachments)
        raise

    persistent_log(
        object_type="letter",
        object_id=letter.id,
        component="body",
        action="upload",
        description="Загрузка тела письма {}".format(letter.code),
        request=request,
    )


def clone_letter(destination_letter, source_letter):
    _validate_destination_letter_is_empty(destination_letter)

    source_html_body = source_letter.html_body
    source_attachments = source_letter.attachments.all()
    attachment_paths = _clone_attachments(source_attachments)
    html_body = _substitute_html_publish_paths_with_cloned(source_html_body, attachment_paths)

    @transaction.atomic
    def db_upload():
        _upload_cloned_attachments(destination_letter, source_attachments, attachment_paths)
        upload_letter_html_body(destination_letter, html_body)
        _clone_letter_meta(destination_letter, source_letter)

    try:
        db_upload()
    except:
        _unpublish_cloned_attachments(attachment_paths)
        raise

    persistent_log(
        object_type="letter",
        object_id=destination_letter.id,
        component="body",
        action="clone",
        description="Загрузка тела письма {}".format(destination_letter.code),
    )


def get_letter_attachment_paths(letter):
    return [attachment.publish_path for attachment in letter.attachments.all()]


def unpublish_letter_attachments_noexcept(attachment_paths):
    for attachment_path in attachment_paths:
        AvatarsPublisher().unpublish(attachment_path)


def check_letter_html_data_len(data_len):
    return data_len <= settings.LETTER_HTML_MAX_LEN


def check_letter_zip_data_len(data_len):
    return data_len <= settings.LETTER_ZIP_MAX_LEN


def _publish_new_attachments(message_attachments):
    for att in message_attachments:
        try:
            att.publish_path = AvatarsPublisher().publish(att.data)
        except:
            _unpublish_new_attachments(message_attachments)
            raise


def _clone_attachments(source_attachments):
    attachment_paths = {}
    for source_attachment in source_attachments:
        try:
            source_attachment_path = source_attachment.publish_path
            cloned_attachment_path = AvatarsPublisher().clone(source_attachment_path)
            attachment_paths[source_attachment_path] = cloned_attachment_path
        except:
            _unpublish_cloned_attachments(attachment_paths)
            raise
    return attachment_paths


def _unpublish_new_attachments(message_attachments):
    for att in message_attachments:
        if hasattr(att, "publish_path"):
            AvatarsPublisher().unpublish(att.publish_path)


def _unpublish_cloned_attachments(attachment_paths):
    for _, cloned_attachment_path in list(attachment_paths.items()):
        AvatarsPublisher().unpublish(cloned_attachment_path)


def _upload_new_attachments(letter, message_attachments):
    for att in message_attachments:
        _upload_attachment(letter, att)


def _upload_cloned_attachments(letter, source_attachments, attachment_paths):
    for source_attachment in source_attachments:
        _upload_cloned_attachment(
            letter, source_attachment, attachment_paths[source_attachment.publish_path]
        )


def _upload_attachment(letter, message_attachment):
    letter_attachment = LetterAttachment(
        letter=letter,
        filename=message_attachment.filename,
        file_size=len(message_attachment.data),
        uri=message_attachment.uri,
        mime_type=message_attachment.mime_type,
        subtype=message_attachment.subtype,
        source_url=message_attachment.absolute_url,
        publish_path=message_attachment.publish_path,
    )
    letter_attachment.save()


def _upload_cloned_attachment(letter, source_attachment, cloned_publish_path):
    letter_attachment = LetterAttachment.objects.get(id=source_attachment.id)
    letter_attachment.id = None
    letter_attachment.letter = letter
    letter_attachment.publish_path = cloned_publish_path
    letter_attachment.save()


def _drop_unused_attachments(letter_attachments, message):
    web_images_paths = set([_get_path(url) for url in get_web_images_urls_from_html(message)])
    for letter_attachment in letter_attachments:
        if letter_attachment.publish_path in web_images_paths:
            continue
        _drop_attachment(letter_attachment)


def _drop_attachment(letter_attachment):
    letter_attachment.unpublish(AvatarsPublisher())
    letter_attachment.delete()


def upload_letter_html_body(letter, html_body):
    """
    Загружает тело письма в объект letter.

    :param letter: объект Letter в базе
    :param html_body: новое тело письма
    """
    if not check_letter_html_data_len(len(html_body)):
        raise ModelRuntimeError("html_body length exceeds limit")
    _check_unsubscribe_link_placeholder(html_body)
    letter.html_body = html_body
    message = test_render(letter)
    template = message.html
    _validate_user_template_variables(template)
    letter.template_meta = _build_template_meta(template)
    letter.save()


def _upload_letter_meta(letter, html_body_file):
    letter.original_letter_content_hash = filehash(html_body_file)
    letter.description = html_body_file.name
    letter.save()


def _clone_letter_meta(destination_letter, source_letter):
    destination_letter.original_letter_content_hash = source_letter.original_letter_content_hash
    destination_letter.description = source_letter.description
    destination_letter.save()


def _check_unsubscribe_link_placeholder(html_body):
    if html_body and not search_unsubscribe_link(html_body):
        raise UnsubscribeLinkNotFoundError()


def _build_template_meta(template):
    template_meta = {}
    template_meta["variables:user"] = template.user_template_variables
    template_meta["variables:builtin"] = template.builtin_template_variables
    return template_meta


def _load_message(letter, html_body_file):
    file_type, message = load_message_from_file(letter=letter, file=html_body_file)
    data_len = len(html_body_file)
    if file_type == "html":
        if not check_letter_html_data_len(data_len):
            raise RuntimeError("letter html size is too long")
    elif file_type == "zip":
        if not check_letter_zip_data_len(data_len):
            raise RuntimeError("letter zip size is too long")
    else:
        logging.warning("Unknown file_type=%s", file_type)
    return message


def _validate_user_template_variables(template):
    user_template_variables = template.user_template_variables
    if len(user_template_variables) > settings.USER_TEMPLATE_VARIABLES_MAX_COUNT:
        raise UserTemplateVariablesCountExceeded()
    for user_template_variable in user_template_variables:
        if len(user_template_variable) > settings.USER_TEMPLATE_VARIABLE_MAX_LENGTH:
            raise UserTemplateVariableLengthExceeded()


def _validate_html_local_images_are_attached(message):
    html_local_images_urls = get_local_images_urls_from_html(message)
    existed_attachments_urls = set(
        [attachment.absolute_url for attachment in _existed_attachments(message.attachments)]
    )
    for html_local_image_url in html_local_images_urls:
        if html_local_image_url not in existed_attachments_urls:
            # Hack for backward compatibility, fix in RTEC-5828
            raise TemplateRuntimeError("Template runtime error:\"'%s'\"" % html_local_image_url)


def _validate_attachments_total_count(letter, message):
    total_count = _new_attachments_count(message) + _reused_attachments_count(
        letter.attachments.all(), message
    )
    if total_count > settings.LETTER_ATTACHMENTS_MAX_COUNT:
        raise AttachmentsCountExceeded()


def _validate_destination_letter_is_empty(destination_letter):
    if len(destination_letter.html_body):
        raise RuntimeError("destination letter should have empty body")
    if len(destination_letter.attachments.all()):
        raise RuntimeError("destination letter should not have attachments")


def _new_attachments_count(message):
    return len(_existed_attachments(message.attachments))


def _reused_attachments_count(letter_attachments, message):
    web_images_paths = set([_get_path(url) for url in get_web_images_urls_from_html(message)])
    reused_attachments = [
        attach for attach in letter_attachments if attach.publish_path in web_images_paths
    ]
    return len(reused_attachments)


def _validate_attachments_total_size(letter, message):
    total_size = _new_attachments_size(message) + _reused_attachments_size(
        letter.attachments.all(), message
    )
    if total_size > settings.LETTER_ATTACHMENTS_MAX_SIZE:
        raise AttachmentsSizeExceeded()


def _new_attachments_size(message):
    size = 0
    for attachment in _existed_attachments(message.attachments):
        size += len(attachment.data)
    return size


def _reused_attachments_size(letter_attachments, message):
    web_images_paths = set([_get_path(url) for url in get_web_images_urls_from_html(message)])
    reused_attachments = [
        attach for attach in letter_attachments if attach.publish_path in web_images_paths
    ]
    return sum([attach.file_size for attach in reused_attachments])


def _substitute_html_local_images_urls_with_published(message):
    substitute_paths = {}
    for attachment in message.attachments:
        publish_url = AvatarsPublisher.get_read_url(attachment.publish_path)
        substitute_paths[attachment.uri] = publish_url
    substitute_images_paths(message, substitute_paths)


def _substitute_html_publish_paths_with_cloned(html_body, attachment_paths):
    html_body = str(html_body)
    for source_attachment_path, attachment_path in list(attachment_paths.items()):
        html_body = html_body.replace(source_attachment_path, attachment_path)
    return html_body


def _existed_attachments(attachments):
    return [attachment for attachment in attachments if _is_attachment_existed(attachment)]


def _is_attachment_existed(attachment):
    if attachment.local_loader is None:
        return False
    return attachment.local_loader[attachment.uri] is not None


def _get_path(url):
    return urlparse(url).path
