import logging
from urllib.parse import urlencode

from fan.emails import Message
from fan.emails.store import BaseFile
from fan.emails.utils import MessageID

from fan.links.unsubscribe import encode_unsubscribe_code2, UNSUBSCRIBE_LINK, get_unsubscribe_links
from fan.utils.so import SO_RESERVED_HEADERS

from .template import Template as T, DummyTemplate
from .template_extensions import RenderFunctions
from .transformer import SendrRegexpTransformer


class SendrBaseMessage(Message):
    transformer_cls = SendrRegexpTransformer


class SendrMessage(SendrBaseMessage):
    RESERVED_HEADERS = {
        "x-yandex-service",
    }

    def __init__(self, letter, skip_image_inline=False):
        self.letter = letter
        self.campaign = letter.campaign
        self.letter_secret = ""
        self._additional_headers = {}

        # В replace_from_message кладём BaseMessage, из которого берём данные на подмену, например, аттачи
        # replace_from_message не храним в кэше
        self.replace_from_message = None

        super().__init__(**self._get_message_params())
        self.load_published_attachments()
        self.after_build = self.after_build_func()
        self.before_build = self.before_build_func()

        # Остается для шаблонов, у которых ссылки на аттачи не обернуты в теги
        if self.letter.html_body and not skip_image_inline:
            self.make_all_images_inline(save=True)

    # to pass widget_headers, so_headers
    def add_headers(self, headers, replace=False, allow_so_headers=False):
        if headers:
            # не даем переписывать заголовки СО, если нет явного разрешения
            headers = {
                k: v
                for k, v in headers.items()
                if allow_so_headers or k.lower() not in SO_RESERVED_HEADERS
            }

        if replace:
            self._additional_headers = headers
        elif headers:
            self._additional_headers.update(headers)

    def reinit(self, **kw):
        # TODO: оторвать этот метод, перевести всё на replace_from_message
        params = self._get_message_params()
        params.update(kw)
        super().__init__(**params)
        self.load_published_attachments()
        self.after_build = self.after_build_func()
        self.before_build = self.before_build_func()

    @property
    def attachments(self):
        return MasterSlaveFilestoreProxy(
            master=super().attachments,
            replace=self.replace_from_message and self.replace_from_message.attachments or None,
        )

    def set_mail_from(self, mail_from):
        super().set_mail_from(mail_from)

    def get_mail_from(self):
        # Если есть replace_from_message и у него установлен mail_from, то отдать его.
        if self.replace_from_message:
            r = self.replace_from_message.get_mail_from()
            if r:
                return r
        return super().get_mail_from()

    mail_from = property(get_mail_from, set_mail_from)

    def get_subject_template_text(self):
        return self._subject.template_text

    def set_subject_template_text(self, template_text):
        self._subject = DummyTemplate(template_text)

    subject_template_text = property(get_subject_template_text, set_subject_template_text)

    @property
    def native_user_template_variables(self):
        subject_variables = set(self._subject.user_template_variables)
        body_variables = set(self.html.user_template_variables)
        return list(subject_variables | body_variables)

    @property
    def user_template_variables(self):
        variables = [variable.lower() for variable in self.native_user_template_variables]
        return list(set(variables))

    def _get_message_params(self):
        letter = self.letter
        headers = {
            "X-Mailer": "YandexSender/0.1",
            "List-ID": "<%s>" % letter.get_magic_list_id(full=True),
            "X-Mailru-Msgtype": "<%s>" % letter.get_magic_list_id(full=False),
            "X-Auto-Response-Suppress": "All",
        }

        if letter.campaign.is_promo:
            headers["Precedence"] = "bulk"

        if letter.reply_to:
            headers["Reply-To"] = letter.reply_to

        # Проставляем метку trust_5 для остальных
        headers[
            "X-Yandex-Hint"
        ] = "bGFiZWw9U3lzdE1ldGthU086dHJ1c3RfNQo="  # 'label=SystMetkaSO:trust_5\n'

        return dict(
            subject=DummyTemplate(letter.subject),
            html=T(letter.html_body),
            mail_from=(letter.from_name, letter.from_email),
            headers=headers,
            attachments=self.load_attachments(),
            message_id=MessageID(),
        )

    def load_attachments(self):
        def _attachment_from_model(obj):
            return BaseFile(
                uri=obj.uri,
                data=obj.data,
                mime_type=obj.mime_type,
                subtype=obj.subtype,
                content_disposition="inline",
            )

        return [
            _attachment_from_model(a)
            for a in self.letter.attachments.filter(publish_path__isnull=True)
        ]

    def load_published_attachments(self):
        self.published_attacments = {}
        for attach in self.letter.attachments.filter(publish_path__isnull=False):
            self.published_attacments[attach.uri] = attach.publish_path

    def set_user_template_variable_values(self, data):
        self.render_data = _transform_keys_to_lower(data)
        for native_variable in self.native_user_template_variables:
            self.render_data[native_variable] = self.render_data.get(native_variable.lower(), "")

    def before_build_func(self):
        """
        1. Установим некоторые переменные. Они нужны для генерации обёрнутых ссылок на статистику.
        2. Положим ссылку на отписку в переменную unsubscribe_link
        """

        def wrapper(message):
            message.render_data["sender_letter_code"] = self.letter.code
            message.render_data["sender_letter_id"] = self.letter.id
            message.render_data["sender_campaign_id"] = self.campaign.id
            if not self.letter_secret:
                logging.warning("No SendrMessage.letter_secret is missing")
            message.render_data["sender_letter_secret"] = self.letter_secret

            if message.campaign.check_unsubscribe:
                self.render_data.update(
                    get_unsubscribe_links(
                        render_context=self.render_data,
                        letter_secret=self.letter_secret,
                        allow_custom=self.letter.campaign.account.allow_custom_unsubscribe_link,
                    )
                )

            # Дополнительные правки в контексте рендеринга писем
            # SENDER-470: При отправке тестовых писем и через списки в контекст попадает поле "email"
            if "email" not in message.render_data:
                message.render_data["email"] = self.mail_to[0][1]
            # SENDER-531: Добавление в шаблон UTM разметки. Используются зарезервированные переменные шаблона
            if self.campaign.use_utm:
                utm_params = {k: v for k, v in self.letter.get_utm().items() if v}
                utm_str = urlencode(utm_params)
                # Параметры UTM добавляются к другим параметрам
                message.render_data["__UTM__"] = "&{}".format(utm_str)
                # Параметры UTM добавляются как единственные параметры
                message.render_data["__UTM_ONLY__"] = "?{}".format(utm_str)

            # Функции для расширений шаблона
            ext_funcs = RenderFunctions(self, self.letter_secret).to_dict()
            message.render_data.update(ext_funcs)

        return wrapper

    def after_build_func(self):
        """
        Добавить:
         * метку яндекс-сервиса X-Yandex-Service
         * заголовок List-Unsubscribe
        """

        def _wrapper(orig, msg):
            unsubscribe_link = orig.render_data.get(UNSUBSCRIBE_LINK)
            if unsubscribe_link:
                msg["List-Unsubscribe"] = unsubscribe_link

            # для баунсов
            msg["X-Sendr-Id"] = self.letter_secret

            if orig._additional_headers:
                for k, v in list(orig._additional_headers.items()):
                    if k.lower() not in self.RESERVED_HEADERS:
                        # Так заменяют заготовки в email.message.Message
                        del msg[k]
                        msg[k] = v

        return _wrapper

    def make_all_images_inline(self, save=False):
        for a in self.attachments:
            a.is_inline = True
        self.transformer.synchronize_inline_images()
        if save:
            self.transformer.save()

    def make_letter_secret(self, email):
        return make_letter_secret(
            email=email,
            letter=self.letter,
            message=self,
            for_testing=False,
        )


def _transform_keys_to_lower(data):
    res = {}
    for key, value in list(data.items()):
        res[key.lower()] = value
    return res


def make_testing_subject(letter, subject):
    campaign = letter.campaign
    if campaign.is_ab:
        suffix = "[TEST:%s:%s]" % (campaign.id, letter.code)
    else:
        suffix = "[TEST:%s]" % campaign.id
    return "%s %s" % (subject, suffix)


def make_letter_secret(email, letter, message, for_testing):
    campaign = letter.campaign
    return encode_unsubscribe_code2(
        email=email,
        campaign_id=campaign.id,
        letter_id=letter.id,
        message_id=message.message_id,
        for_testing=for_testing,
    )


class MasterSlaveFilestoreProxy:
    def __init__(self, master, replace):
        self.master = master
        self.replace = replace

    def __iter__(self):
        cache = set()
        if self.replace:
            # Временные аттачи должны перезаписать одноимённые постоянные аттачи шаблона
            for att in self.replace:
                cache.add(att.uri)
                yield att

        for att in self.master:
            if att.uri not in cache:
                yield att

    def add(self, *args, **kwargs):
        self.master.add(*args, **kwargs)
