import urllib.request, urllib.parse, urllib.error
import urllib.parse

from django.conf import settings

import pytz
from jinja2 import nodes
from jinja2.exceptions import TemplateRuntimeError, UndefinedError
from jinja2.ext import Extension

from fan.file_models import AvatarsPublisher
from fan.utils.date import Now
import collections


# Новый формат настроек пикселя и обертки ссылок для расширений шаблона
SENDR_TEMPLATE_EXTENSION_PIXEL = "{BASE_URL}/px/{CAMPAIGN_ID}/{LETTER_ID}/{LETTER_SECRET}"
SENDR_TEMPLATE_EXTENSION_LINK_PREFIX = (
    "{BASE_URL}/l/{CAMPAIGN_ID}/{LETTER_ID}/{LINK_ID}/{LETTER_SECRET}/*"
)


class BaseSingleBlockExtension(Extension):
    """
    Базовый класс для реализации тегов в один блок ({%<tag> [<param>]...%})
    """

    tags = frozenset()  # Теги, на которые реагирует расширение
    end_tags = {}  # Если блоку нужен закрывающий тег, то ему здесь самое место

    def parse(self, parser):
        tag_node = next(parser.stream)
        tag_name = tag_node.value
        lineno = tag_node.lineno
        if parser.stream.current.type != "block_end":
            param = parser.parse_tuple()
            if isinstance(param, nodes.Tuple):
                params = param.items
            else:
                params = [param]
        else:
            params = []

        end_tag = self.end_tags and self.end_tags.get(tag_name)
        if end_tag:
            body = parser.parse_statements(["name:{}".format(end_tag)], drop_needle=True)
        else:
            body = ""

        ctx_node = nodes.ContextReference()
        tag_name_node = nodes.Const(tag_name)

        render_args = [ctx_node, tag_name_node] + params
        return nodes.CallBlock(
            self.call_method("_render", args=render_args), (), (), body
        ).set_lineno(lineno)

    def _render(self, ctx, tag, *args, **kwargs):
        raise NotImplementedError

    @staticmethod
    def get_stat_function(stat_dict):
        """
        Получение замыканий для сбора статистики использования расширения в шаблоне
        Статистика должна добавляться в stat_dict при вызове функции
        """
        raise NotImplementedError


class NowTag(BaseSingleBlockExtension):
    """
    Render current datetime with given format string in given timezone.

    If tag used without arguments: {% now %}, it renders to
    >>>now(tz='Europe/Moscow').isoformat(sep=' ')

    First optional argument can be format string:
    {% now "current time is %H:%M:%S" %} renders to
    >>>now(tz='Europe/Moscow').strftime("current time is %H:%M:%S")

    Second optional argument is timezone name to localize current time:
    {% now "current time is %H:%M:%S", "Europe/Paris" %} renders to
    >>>now(pytz.timezone("Europe/Paris")).strftime("current time is %H:%M:%S")
    """

    tags = frozenset(["now"])

    def _get_tz_info(self, tz_name):
        try:
            return pytz.timezone(tz_name)
        except pytz.UnknownTimeZoneError:
            raise TemplateRuntimeError("Bad timezone argument in {% now %} tag.")

    def _render(self, ctx, tag, *args, **kwargs):
        if args:
            fmt = str(args[0])
        else:
            fmt = None

        if len(args) > 1:
            tz_name = str(args[1])
            tz_info = self._get_tz_info(tz_name)
        else:
            tz_info = Now.MOSCOW_TZ

        dt_now = Now(tz=tz_info)

        # any string is ok as strftime argument
        if fmt:
            return dt_now().strftime(fmt)
        else:
            return dt_now().isoformat(sep=str(" "))

    @staticmethod
    def get_stat_function(stat_dict):
        return {}


class OpenCounterTag(BaseSingleBlockExtension):
    tags = frozenset(["opens_counter"])

    def _render(self, ctx, tag, *args, **kwargs):
        pixel_url_func = ctx.resolve("_get_pixel_url")
        if not pixel_url_func or not isinstance(pixel_url_func, collections.Callable):
            raise UndefinedError("Function _get_pixel_url should be in environment")

        pixel_url = pixel_url_func()
        if pixel_url:
            return '<img src="%s" width="1" height="1" />' % pixel_url
        else:
            return ""

    @staticmethod
    def get_stat_function(stat_dict):
        def _get_pixel_url():
            stat_dict.setdefault("pixel_usage", 0)
            stat_dict["pixel_usage"] += 1
            return ""

        return {"_get_pixel_url": _get_pixel_url}


class WrapLinkTag(BaseSingleBlockExtension):
    tags = frozenset(["wrap"])
    end_tags = {"wrap": "endwrap"}

    def _render(self, ctx, tag, *args, **kwargs):

        id_ = args and args[0] or None
        caller = kwargs.get("caller")
        link = caller and caller() or None

        link_url_func = ctx.resolve("_wrap_link_url")
        if not link_url_func or not isinstance(link_url_func, collections.Callable):
            raise UndefinedError("Function _wrap_link_url should be in environment")
        return link_url_func(id_, link)

    @staticmethod
    def get_stat_function(stat_dict):
        def _wrap_link_url(id_, link):
            if id_:
                stat_dict.setdefault("wrapped_links", [])
                stat_dict["wrapped_links"].append((id_, link))
            return ""

        return {"_wrap_link_url": _wrap_link_url}


class SharedUrl(BaseSingleBlockExtension):
    tags = frozenset(["autoloaded_file"])

    def _render(self, ctx, tag, link, *args, **kwargs):
        uploaded_url_func = ctx.resolve("_get_uploaded_url")
        if not uploaded_url_func or not isinstance(uploaded_url_func, collections.Callable):
            raise UndefinedError("Function _get_uploaded_url should be in environment")
        return uploaded_url_func(link, tag=tag)

    @staticmethod
    def get_stat_function(stat_dict):
        def _get_uploaded_url(link, tag=None):
            if link and tag:
                if tag == "autoloaded_file":
                    stat_dict.setdefault("attachments", [])
                    stat_dict["attachments"].append(link)
            return ""

        return {"_get_uploaded_url": _get_uploaded_url}


class RenderFunctions:
    """
    Класс, предоставляющий функции для рендеринга расширений в письме
    """

    _STATS_DEFAULTS = getattr(settings, "SENDR_STATS_DEFAULTS", {})
    _PIXEL_PATTERN = SENDR_TEMPLATE_EXTENSION_PIXEL
    _LINK_PATTERN = SENDR_TEMPLATE_EXTENSION_LINK_PREFIX
    _UTM_PARAMS = (
        "utm_source",
        "utm_medium",
        "utm_campaign",
        "utm_content",
    )
    _UTM_EXCEPT_DOMAINS = getattr(settings, "SENDR_UTM_DONT_WRAP", ())

    def __init__(self, message, secret, wrap_links=False):
        self.base_url = self._STATS_DEFAULTS.get("BASE_URL")
        self.message = message
        self.campaign = message.campaign
        self.letter = message.letter
        self.secret = secret
        self.add_utm = self.campaign.use_utm
        self.wrap_links = wrap_links
        self.utm = self.letter.get_utm() if self.add_utm else None

    def _get_pixel_func(self):
        def _get_pixel_url():
            return self._PIXEL_PATTERN.format(
                BASE_URL=self.base_url,
                CAMPAIGN_ID=self.campaign.id,
                LETTER_ID=self.letter.id,
                LETTER_SECRET=self.secret,
            )

        return _get_pixel_url

    def _get_link_wrap(self):
        link_wrap_prefix = self._LINK_PATTERN.format(
            BASE_URL=self.base_url,
            CAMPAIGN_ID=self.campaign.id,
            LETTER_ID=self.letter.id,
            LINK_ID="{id}",
            LETTER_SECRET=self.secret,
        )

        def _wrap_link_url(id_, link):
            # urllib плохо работает с юникодом, поэтому преобразуем в строку
            if isinstance(link, str):
                link = link.encode("utf-8")

            if id_ and self.wrap_links:
                id_string = id_.encode("utf-8") if isinstance(id_, str) else str(id_)
                prefix = link_wrap_prefix.format(id=urllib.parse.quote(id_string)).encode()
            else:
                prefix = b""
            if self.add_utm:
                parsed_link = urllib.parse.urlparse(link)
                if parsed_link.netloc.lower() not in self._UTM_EXCEPT_DOMAINS:
                    qs = urllib.parse.parse_qsl(parsed_link.query)

                    link_utm = set((k.lower() for k, _ in qs if k.lower() in self._UTM_PARAMS))
                    additional_utm = []

                    for k in self._UTM_PARAMS:
                        if k in link_utm:
                            continue
                        utm_value = self.utm.get(k)
                        if utm_value:
                            additional_utm.append((k, utm_value))
                    qs.extend(additional_utm)
                    parsed_link = list(parsed_link)
                    parsed_link[4] = urllib.parse.urlencode(qs)
                    link = urllib.parse.urlunparse(parsed_link)

            # если в обработке url остался строкой, то возвращаем в unicode явно, иначе падает на кириллице
            if isinstance(link, str):
                link = link.decode("utf-8")

            return prefix + link

        return _wrap_link_url

    def _get_image_urls(self):
        unpublished = {}
        unpublished_by_filename = {}
        for a in self.message.attachments:
            unpublished[a.uri] = a.content_id
            unpublished_by_filename[a.filename] = a.content_id
        published = self.message.published_attacments

        def _get_uploaded_url(link, tag):
            if tag == "autoloaded_file":
                if link in unpublished:
                    return "cid:" + unpublished[link]
                if link in unpublished_by_filename:
                    return "cid:" + unpublished_by_filename[link]
                else:
                    return AvatarsPublisher.get_read_url(published[link])

        return _get_uploaded_url

    def to_dict(self):
        return {
            "_get_pixel_url": self._get_pixel_func(),
            "_wrap_link_url": self._get_link_wrap(),
            "_get_uploaded_url": self._get_image_urls(),
        }


registered = (NowTag, OpenCounterTag, WrapLinkTag, SharedUrl)

used_tags = set(["raw", "endraw"])
for ext in registered:
    for tag in ext.tags:
        used_tags.add(tag)
    for tag in list(ext.end_tags.values()):
        used_tags.add(tag)

used_tags = frozenset(used_tags)
