import re

from fan.emails.transformer import BaseTransformer, MessageTransformer


class Element:
    def __init__(self, uri):
        self.uri = uri


class RegexpTransformerMixin:
    LINK_REGEXP = re.compile(
        r'(<(?:a|v[:]\S+)\s.*?href\s*=\s*[\'"]?\s*)([^><"\']*)(\s*[\'"]?.*?>)',
        re.UNICODE | re.MULTILINE | re.DOTALL | re.IGNORECASE,
    )
    IMG_REGEXP = re.compile(
        r'(<(?:img|v[:]\S+)\s.*?src\s*=\s*[\'"]?\s*)([^><"\']*)(\s*[\'"]?.*?>)',
        re.UNICODE | re.MULTILINE | re.DOTALL | re.IGNORECASE,
    )
    BACKGROUND_REGEXP = re.compile(
        r"(<\w+\s[^<>]*background\s*=\s*\"\s*)([^\"<>]*)(\s*\".*?>)",
        re.UNICODE | re.MULTILINE | re.DOTALL | re.IGNORECASE,
    )
    STYLE_REGEXP = re.compile(
        r"<style>[^<>]*</style>", re.UNICODE | re.MULTILINE | re.DOTALL | re.IGNORECASE
    )
    INLINE_STYLE_REGEXP = re.compile(
        r'(<[^<>]+style\s*=\s*)(?P<sqote>["\'])(.*?)((?P=sqote))',
        re.UNICODE | re.MULTILINE | re.DOTALL | re.IGNORECASE,
    )
    URL_REGEXP = re.compile(
        r'(url\([\'"]?)([^\(\)\'"]*)([\'"]?\))',
        re.UNICODE | re.MULTILINE | re.DOTALL | re.IGNORECASE,
    )
    PIXEL_TAG_REGEXP = re.compile(
        r'<img src="https*://.+/px/{{ sender_campaign_id }}/{{ sender_letter_id }}/{{ sender_letter_secret }}" width="1" height="1" />',
        re.UNICODE | re.IGNORECASE,
    )

    _html = None

    def attribute_value(self, el):
        return None

    def apply_to_links(self, func):
        def repl(m_obj):
            return "".join((m_obj.group(1), func(m_obj.group(2)) or "", m_obj.group(3)))

        self._html = self.LINK_REGEXP.sub(repl, self._html)

    def apply_to_images(self, func, images=True, backgrounds=True, styles_uri=True):
        def repl(m_obj):
            if m_obj.group(2).strip():
                return "".join(
                    (
                        m_obj.group(1),
                        func(m_obj.group(2), element=Element(m_obj.group(2))) or "",
                        m_obj.group(3),
                    )
                )
            else:
                return "".join((m_obj.group(1), m_obj.group(2), m_obj.group(3)))

        def style_repl(m_obj):
            return self.URL_REGEXP.sub(repl, m_obj.group(0))

        def inline_style_repl(m_obj):
            groups = list(m_obj.groups())
            groups[-2] = self.URL_REGEXP.sub(repl, groups[-2])
            return "".join(groups)

        if images:
            self._html = self.IMG_REGEXP.sub(repl, self._html)

        if backgrounds:
            # Apply to images from <tag background="X">
            self._html = self.BACKGROUND_REGEXP.sub(repl, self._html)

        if styles_uri:
            self._html = self.STYLE_REGEXP.sub(style_repl, self._html)
            self._html = self.INLINE_STYLE_REGEXP.sub(inline_style_repl, self._html)

    def append_to_body(self, el):
        splited_body = self._html.split("</body>")
        splited_body[0] += el

        self._html = "</body>".join(splited_body)

    def replace_old_pixel(self):
        self._html = self.PIXEL_TAG_REGEXP.sub("{% opens_counter %}", self._html)

    def numerate_links(self, attr="data-link-number"):
        links = {}

        def repl(m_obj):
            url = m_obj.group(2)
            if "*" in url:
                prefix, clean_url = url.split("*", 1)
                pref_parts = prefix.rsplit("/", 3)
                if len(pref_parts) >= 2:
                    id_ = pref_parts[1]
                    tag = '{head}{{% wrap "{id}" %}}{url}{{% endwrap %}}{tail}'.format(
                        id=id_, url=clean_url, head=m_obj.group(1), tail=m_obj.group(3)
                    )
                    return tag
            return "".join(m_obj.groups())

        self._html = self.LINK_REGEXP.sub(repl, self._html)
        return links

    def to_string(self, *args, **kwargs):
        return self._html


class SendrRegexpTransformer(RegexpTransformerMixin, MessageTransformer):
    def _load_attachment_func(self, uri, element=None, callback=None, **kw):

        if callback is None:
            # Default callback: skip images with data-emails="ignore" attribute
            def callback(_, hints):
                return hints["attrib"] != "ignore"

        attribute_value = self.attribute_value(element) or ""

        # If callback returns False, skip attachment loading
        if not callback(element, hints={"attrib": attribute_value}):
            return uri

        attachment = self.attachment_store.by_uri(uri)
        if attachment is None:
            attachment = self.attachment_file_cls(
                uri=uri,
                absolute_url=self.get_absolute_url(uri),
                local_loader=self.local_loader,
                content_disposition="inline" if "inline" in attribute_value else None,
                requests_args=self.requests_params,
            )
            self.attachment_store.add(attachment)
        return attachment.uri


class HtmlRegexpTransformer(RegexpTransformerMixin, BaseTransformer):
    pass
