from django.contrib.postgres.fields import ArrayField
from django.core.exceptions import ObjectDoesNotExist
from django.db import models
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from fan.db.fields.separatedvaluesfield import SeparatedValuesField

from .campaign import Campaign
from .common import TimestampsMixin


"""
Аккаунт -> Проект -> Кампания
"""


class Account(TimestampsMixin, models.Model):
    """
    Аккаунт заказчика.
    Например, "Яндекс-Почта" - это аккаунт.
    """

    org_id = models.CharField(max_length=256, db_index=True)
    name = models.SlugField(unique=True, max_length=192, db_index=False)
    title = models.CharField(max_length=256)
    from_logins = ArrayField(models.CharField(max_length=256), default=list)
    default_testing_emails = SeparatedValuesField(max_length=1024, null=True, blank=True)

    # lightweight alternative to another Option
    allow_custom_unsubscribe_link = models.BooleanField(
        default=False,
        help_text="Разрешить аккаунту задавать свою ссылку отписки в контексте рендеринга письма",
    )

    # время хранения статусов отправки в минутах. Актуально для транзакционных рассылок
    status_store_period = models.IntegerField(
        default=0, verbose_name="Время хранения статусов отправки"
    )

    class Meta:
        verbose_name_plural = _("Сервис")

    def __str__(self):
        return self.name

    def get_campaign(
        self, id=None, api_code=None, strict=False, on_not_found=None, exclude_deleted=True
    ):

        # TODO: заменить на Campaign.objects.get_campaign

        if id:
            spec = {"id": id}
        elif api_code:
            spec = {"api_code": api_code}
        else:
            return None

        if exclude_deleted:
            spec["deleted"] = False

        try:
            o = Campaign.objects.get(account=self, **spec)
        except ObjectDoesNotExist:
            if strict:
                if on_not_found:
                    raise on_not_found()
                else:
                    raise
            o = None

        return o

    def get_default_project(self):
        return self.project_set.first()

    def get_title(self):
        return self.title or self.name

    def has_access(self, user, access=None):
        queryset = self.member_set.filter(user=user)
        if access is not None:
            queryset = queryset.filter(type__lte=access)
        return queryset.exists()

    def add_member(self, **kwargs):
        self.member_set.get_or_create(**kwargs)
        # TODO: оторвать со временем
        self.roles.get_or_create(**kwargs)

    def get_or_create_default_key(self):
        return self.key_set.get_or_create_key()

    def get_active_key(self):
        return self.key_set.get_active_key()

    @cached_property
    def local_moderators(self):
        """
        Возвращаем список пользователей-модераторов в этом аккаунте.
        Не учитываем глобальных модераторов.

        :return: generator([User, ...])
        """
        from . import UserRole

        for r in UserRole.objects.filter(account=self, role=UserRole.ROLES.MODERATOR):
            yield r.user

    def get_domain(self):
        return self.unsub_domains.first()

    def save(self, *args, **kwargs):
        super().save(*args, **kwargs)
        from .unsubscribe import AccountUnsubscribeList

        if not self.unsubscribe_lists.exists():
            AccountUnsubscribeList.objects.create_default(account=self)
            AccountUnsubscribeList.objects.create_general(account=self)
