from django.core.exceptions import ObjectDoesNotExist
from django.db import models
from django.db.models import Q
from django.utils.translation import ugettext_lazy as _
from jsonfield import JSONField
from fan.db.fields.separatedvaluesfield import SeparatedValuesField
from fan.db.fields.slug import SlugDotField
from fan.db.lookups import ILike  # NOQA
from fan.delivery.status import DeliveryStatus
from fan.utils.base_n import to_base26

from .common import ENV_SUFFIX, TimestampsMixin
from .letter import Letter
from .maillist import SingleUseMailList

TITLE_FILTER_MODE_EXACT = "exact"
TITLE_FILTER_MODE_OR = "or"
TITLE_FILTER_MODE_AND = "and"


class CampaignQuerySet(models.QuerySet):
    def filter_by_title(self, search_string, mode=TITLE_FILTER_MODE_AND):
        """
        Given string filter campaigns by title.
        Three modes available:
            and: split search string into words by spaces, then search titles containing all of the words
            or: split search string into words by spaces, then search titles containing at least one word
            exact: search string is contained in title

        All checks are case insensitive.
        """
        if mode in (TITLE_FILTER_MODE_AND, TITLE_FILTER_MODE_OR):
            q = Q()
            key_words = [word for word in search_string.strip().split(" ") if word]
            for word in key_words:
                cond = Q(name__ilike=word)
                if mode == TITLE_FILTER_MODE_OR:
                    q |= cond
                else:
                    q &= cond
        elif mode == TITLE_FILTER_MODE_EXACT:
            q = Q(name__ilike=search_string)
        else:
            q = Q()

        return self.filter(q)

    def exclude_deleted(self):
        # not nullable поле, фильтруем по явному значению, чтобы работали индексы
        return self.filter(deleted=False)


class CampaignManager(models.Manager):
    def get_queryset(self):
        return CampaignQuerySet(self.model, using=self._db, hints=self._hints)

    def get_campaign(
        self,
        id=None,
        account=None,
        strict=False,
        on_not_found=None,
        exclude_deleted=True,
    ):
        if id:
            spec = {"id": id}
        else:
            return None

        if exclude_deleted:
            spec["deleted"] = False

        try:
            o = self.get(**spec)
        except ObjectDoesNotExist:
            if strict:
                if on_not_found:
                    raise on_not_found()
                else:
                    raise
            o = None
        return o

    def filter_by_title(self, search_string, mode="and"):
        return self.get_queryset().filter_by_title(search_string, mode=mode)

    def exclude_deleted(self):
        # not nullable поле, фильтруем по явному значению, чтобы работали индексы
        return self.get_queryset().filter(deleted=False)


class Campaign(TimestampsMixin, models.Model):
    """
    Это одна "рассылка" - собственно письмо рассылки
    """

    objects = CampaignManager()

    # Глобальные статусы
    STATUS_EMBRYON = "embryon"
    STATUS_DRAFT = "draft"
    STATUS_ONREVIEW = "onreview"
    STATUS_SENDING = "sending"
    STATUS_DONE = "done"
    STATUS_SENT = "sent"
    STATUS_FAILED = "failed"
    STATUS_CANCELING = "canceling"
    STATUS_CANCELED = "canceled"

    STATUS_ACTIVE = "active"  # для транзакционных
    STATUS_DISABLED = "disabled"  # для транзакционных

    GLOBAL_STATE_CHOICES = (
        (STATUS_EMBRYON, _("Еще не создана")),
        (STATUS_DRAFT, _("На редактировании")),
        (STATUS_ONREVIEW, _("На утверждении")),
        (STATUS_ACTIVE, _("Активна")),
        (STATUS_DISABLED, _("Остановлена")),
        (STATUS_SENDING, _("На отправке")),
        (STATUS_DONE, _("Завершена")),
        (STATUS_SENT, _("Отправлена")),
        (STATUS_FAILED, _("Сбой отправки")),
        (STATUS_CANCELING, _("Отменяется")),
        (STATUS_CANCELED, _("Отменена")),
    )

    VALID_GLOBAL_STATES = set([choice[0] for choice in GLOBAL_STATE_CHOICES])

    # Статусы модерации
    MODERATE_STATE_SUBMITTED = "submitted"
    MODERATE_STATE_APPROVED = "approved"
    MODERATE_STATE_DECLINED = "declined"

    MODERATE_STATE_CHOICES = (
        (None, _("Не готова")),
        (MODERATE_STATE_SUBMITTED, _("Отправлена на утверждение")),
        (MODERATE_STATE_APPROVED, _("Утверждена")),
        (MODERATE_STATE_DECLINED, _("Отклонена")),
    )

    # Статусы отправки
    SEND_STATE_SENDING_AB = "sending_ab"  # для АБ-рассылки: фаза 1 - отправка вариантов А/Б
    SEND_STATE_SENT_AB = "sent_ab"  # для АБ-рассылки: фаза 1 - отправка вариантов А/Б
    SEND_STATE_SENDING = "sending"  # для обычной рассылки: отправка; для АБ-рассылки: отправка победителя (вторая фаза)
    SEND_STATE_SENT = "sent"  # отправка завершена

    SEND_STATE_CHOICES = (
        (None, _("Еще не отправлялась")),
        (SEND_STATE_SENDING_AB, _("Отправляются варианты А/Б")),
        (SEND_STATE_SENT_AB, _("Отправлены варианты А/Б")),
        (SEND_STATE_SENDING, _("Отправляется")),
        (SEND_STATE_SENT, _("Отправлена")),
    )

    SEND_CONTROL_PAUSE = "pause"
    SEND_CONTROL_CANCEL = "cancel"

    SEND_CONTROL_CHOICES = (
        (None, _("Всё идёт по плану")),
        # или остановлена, если речь идет про транзакционные рассылки
        (SEND_CONTROL_CANCEL, _("Отменена")),
    )

    # Тип рассыки
    TYPE_SIMPLE = "simple"
    TYPE_AB = "AB"
    TYPE_TRANSACTIONAL = "transact"
    TYPE_TRANSACTIONAL_REVISION = "transrev"
    TYPE_PERIODIC = "periodic"
    TYPE_PERIODIC_CHILD = "per_chld"
    TYPE_CHOICES = (
        (TYPE_SIMPLE, _("Промо-рассылка")),
        (TYPE_AB, _("A/B промо-рассылка")),
        (TYPE_TRANSACTIONAL, _("Транзакционное письмо")),
        (TYPE_TRANSACTIONAL_REVISION, _("Новая версия транзакционного письма")),
        (TYPE_PERIODIC, _("Переодические рассылки")),
        (TYPE_PERIODIC_CHILD, _("Запланированная периодическая рассылка")),
    )

    account = models.ForeignKey("Account", related_name="campaign_set")

    # Статусы планирования
    SCHEDULING_WAITING = "waiting"
    SCHEDULING_IN_PROGRESS = "in_progress"
    SCHEDULING_FINISHED = "finished"
    SCHEDULING_SEGMENTS_COUNT = "segments_count"

    SCHEDULING_CHOICES = (
        (SCHEDULING_WAITING, _("Ожидает планирования")),
        (SCHEDULING_IN_PROGRESS, _("Планируется")),
        (SCHEDULING_FINISHED, _("Запланировано")),
        (SCHEDULING_SEGMENTS_COUNT, _("Расчет сегментов")),
    )

    project = models.ForeignKey("Project", related_name="campaigns", null=True)

    magic_list_id = models.CharField(
        _("Магическое слово, используется в статистике постмастера"),
        max_length=128,
        null=True,
        blank=True,
    )
    type = models.CharField(max_length=8, choices=TYPE_CHOICES, default=TYPE_SIMPLE)

    # Глобальный статус (на редактировании, на отправке, ...)
    state = models.CharField(max_length=16, choices=GLOBAL_STATE_CHOICES, default="draft")

    description = models.TextField(_("Описание рассылки, для своих"), blank=True)
    testing_emails = SeparatedValuesField(max_length=10 * 1024, null=True, blank=True)
    testing_emails_text = models.CharField(max_length=10 * 1024, null=True, blank=True)

    use_utm = models.BooleanField(
        default=False, blank=True, help_text=_("Использовать UTM разметку для кампании")
    )

    yandex_metrika = JSONField(
        null=True, blank=True, help_text=_("Счетчики Яндекс.Метрики"), default=dict
    )
    appmetrica = JSONField(null=True, blank=True, help_text=_("События AppMetrica"), default=dict)

    deleted = models.BooleanField(default=False, blank=True)
    created_by = models.CharField(
        max_length=64, editable=False, null=True, blank=True, db_index=True
    )

    # Статус модерации
    # TODO: убрать поля, специфичные для промо-рассылок в отдельную таблицу типа PromoCampaign
    submitted_at = models.DateTimeField(editable=False, null=True)
    submitted_by = models.CharField(
        max_length=64, editable=False, null=True, blank=True, db_index=True
    )
    moderation_state = models.CharField(
        max_length=16, choices=MODERATE_STATE_CHOICES, default=None, blank=True, null=True
    )

    # Когда отправлять
    scheduled_for = models.DateTimeField(null=True, blank=True)

    # Статус отправки
    sending_state = models.CharField(
        max_length=16, choices=SEND_STATE_CHOICES, default=None, blank=True, null=True
    )
    sending_state_modified_at = models.DateTimeField(null=True, blank=True)

    # В sending_control пишем команды для воркеров - поставить рассылку на паузу или остановить
    sending_control = models.CharField(
        max_length=16, choices=SEND_CONTROL_CHOICES, default=None, null=True, blank=True
    )

    # Победитель А/Б тестирования (код письма)
    ab_winner_code = models.CharField(max_length=4, default=None, null=True, blank=True)

    # Если это клон, то клон какой рассылки
    clone_of = models.ForeignKey(
        "self", related_name="clones", null=True, db_constraint=False, blank=True
    )

    ignore_unsubscribe = models.BooleanField(
        default=False, help_text=_("Игнорировать отписки"), db_column="ignore_global_unsubscribe"
    )

    valid_until = models.DateTimeField(
        null=True, help_text=_("После этого времени не отправлять письма"), blank=True
    )

    slug = SlugDotField(unique=False, max_length=192, db_index=False)
    name = models.CharField(max_length=128, default=None, blank=True, null=True)

    # Статус планирования рассылки
    scheduling_status = models.CharField(
        max_length=16, choices=SCHEDULING_CHOICES, default=None, null=True
    )

    maillist = models.ForeignKey(
        "Maillist", related_name="campaigns", on_delete=models.SET_NULL, null=True
    )

    # Списки отписки
    unsubscribe_lists = models.ManyToManyField(
        "AccountUnsubscribeList",
        verbose_name="Списки отписки",
        related_name="campaigns",
        blank=True,
    )

    lawyer_approval_ticket_link = models.CharField(
        max_length=1024,
        default="",
        blank=True,
        verbose_name="Ссылка на тикет согласования с юристом",
    )
    policy_approval_ticket_link = models.CharField(
        max_length=1024,
        default="",
        blank=True,
        verbose_name="Ссылка на тикет согласования с политикой рассылок",
    )

    # control group percent (relatively to maillist size)
    control_group_percent = models.FloatField(
        default=0.0, verbose_name="Процент контрольной группы"
    )

    data = JSONField(null=True, help_text=_("Вспомогательные атрибуты"), default=dict)

    def __str__(self):
        return "%s:%s" % (self.account, self.id)

    def save(self, *args, **kwargs):
        if self.project:
            self.account = self.project.account
        if not self.slug:
            self.slug = self.create_slug()

        super().save(*args, **kwargs)

        if not self.is_transactional and not self.unsubscribe_lists.exists():
            # У промо рассылок всегда должен быть список отписки!!!
            from .unsubscribe import AccountUnsubscribeList

            try:
                default_list = AccountUnsubscribeList.objects.get_default(self.account)
            except ObjectDoesNotExist:
                default_list = AccountUnsubscribeList.objects.create_default(self.account)

            self.unsubscribe_lists.add(default_list)

    @property
    def is_ab(self):
        return self.type == Campaign.TYPE_AB

    @property
    def is_transactional(self):
        return self.type in (Campaign.TYPE_TRANSACTIONAL, Campaign.TYPE_TRANSACTIONAL_REVISION)

    @property
    def is_promo(self):
        return self.type in (Campaign.TYPE_SIMPLE, Campaign.TYPE_AB, Campaign.TYPE_PERIODIC_CHILD)

    @property
    def next_letter_code(self):
        # Человеко-читаемый код письма внутри одной компании: A, B, C, D, ...
        # Нужен для А/Б рассылок
        codes = self.letters.values_list("code", flat=True)
        new_code = len(codes)
        while to_base26(new_code) in codes:
            new_code += 1
        return to_base26(new_code)

    def create_letter(self, **kw):
        letter = Letter(campaign=self, code=self.next_letter_code, revision=0, **kw)
        letter.save()
        return letter

    @property
    def from_email(self):
        return self.default_letter.from_email

    @property
    def from_name(self):
        return self.default_letter.from_name

    @property
    def subject(self):
        return self.default_letter.subject

    @property
    def title(self):
        if self.name:
            return self.name
        letter = self.default_letter
        return letter and letter.subject or "unknown"

    @property
    def letter_uploaded(self):
        return self.default_letter.uploaded

    @property
    def letter_description(self):
        return self.default_letter.description

    @property
    def maillist_uploaded(self):
        if self.maillist:
            return True
        try:
            ml = self.single_use_maillist
        except SingleUseMailList.DoesNotExist:
            return False
        return True

    @property
    def maillist_description(self):
        if self.maillist:
            return self.maillist.filename
        return self.single_use_maillist.description if self.maillist_uploaded else ""

    @property
    def maillist_preview(self):
        if self.maillist:
            return self.maillist.preview
        return self.single_use_maillist.data_preview if self.maillist_uploaded else []

    @property
    def maillist_slug(self):
        if self.maillist:
            return self.maillist.slug
        return ""

    @property
    def maillist_title(self):
        if self.maillist:
            return self.maillist.title
        return ""

    @property
    def maillist_size(self):
        if self.maillist:
            return self.maillist.size
        return self.estimated_subscribers_number()

    @property
    def maillist_created_at(self):
        if self.maillist:
            return self.maillist.created_at
        return self.single_use_maillist.created_at if self.maillist_uploaded else ""

    @property
    def maillist_modified_at(self):
        if self.maillist:
            return self.maillist.modified_at
        return self.single_use_maillist.modified_at if self.maillist_uploaded else ""

    @property
    def stats(self):
        if self.state != Campaign.STATUS_SENT:
            return None
        delivery_stats = {stat.status: stat.count for stat in self.deliveryerrorstats_set.all()}
        campaign_stat = self.stats_rel if hasattr(self, "stats_rel") else None
        return {
            "uploaded": delivery_stats.get(DeliveryStatus.EMAIL_UPLOADED, 0),
            "unsubscribed_before": delivery_stats.get(DeliveryStatus.EMAIL_UNSUBSCRIBED, 0),
            "duplicated": delivery_stats.get(DeliveryStatus.EMAIL_DUPLICATED, 0),
            "invalid": delivery_stats.get(DeliveryStatus.EMAIL_INVALID, 0),
            "unsubscribed_after": getattr(campaign_stat, "unsubscribes", 0) or 0,
            "views": getattr(campaign_stat, "reads", 0) or 0,
        }

    @property
    def default_letter(self):
        all_letters = self.letters_rel.all()
        if all_letters:
            return min(all_letters, key=lambda letter: letter.id)

    def _get_magic_list_id(self):
        r = self.magic_list_id
        if not r:
            r = "sendr-%s" % self.id
            if ENV_SUFFIX:
                r += ENV_SUFFIX
        return r

    def get_letter(self, code=None, id=None, strict=False):
        from fan.utils.getters import _abstract_getter

        spec = {"campaign_id": self.pk}
        if code:
            spec.update(
                {"fields": ["code"], "value": code}
            )  # , 'revision': self.draft_revision}) # при поиске по коду - используем номер ревизии
        else:
            spec.update({"fields": ["id"], "value": id})
        return _abstract_getter(qs=self.letters, strict=strict, **spec)

    @property
    def letters(self):
        return self.letters_rel.order_by("id")

    def estimated_subscribers_number(self):
        try:
            subscribers_number = self.single_use_maillist.subscribers_number
        except ObjectDoesNotExist:
            subscribers_number = 0
        if subscribers_number is None:
            raise Exception("subscribers_number is None")
        return subscribers_number

    @staticmethod
    def create_slug():
        from fan.utils.slug import create_slug

        return create_slug()

    class Meta:
        verbose_name_plural = _("Рассылка")
        unique_together = (("account", "slug"),)

    @property
    def check_unsubscribe(self):
        """
        Проводить ли проверку списков отписки
        """
        from fan.utils.cached_getters import get_campaign_unsubscribe_list_exists

        if self.is_transactional:
            return get_campaign_unsubscribe_list_exists(self.id)

        return True
