from urllib.parse import urlparse

import jsonfield
from django.core.exceptions import ObjectDoesNotExist
from django.db import models
from django.utils.translation import ugettext_lazy as _
from .common import TimestampsMixin


class Letter(TimestampsMixin, models.Model):
    """
    Это одно "письмо" рассылки.
    У рассылки может быть несколько писем:
     а) у А/Б рассылок - варианты A, B, и тп
     б) у транзакционных писем - ревизии
    """

    # Cтатусы письма для ревизий транзакционных рассылок
    STATUS_DRAFT = "draft"
    STATUS_ONREVIEW = "onreview"
    STATUS_ACTIVE = "active"
    STATUS_DISABLED = "disabled"

    STATE_CHOICES = (
        (STATUS_DRAFT, _("На редактировании")),
        (STATUS_ONREVIEW, _("На утверждении")),
        (STATUS_ACTIVE, _("Активна")),
        (STATUS_DISABLED, _("Остановлена")),
    )

    # Статус (активна ли эта ревизия транзакционной рассылки)
    state = models.CharField(max_length=16, choices=STATE_CHOICES, default=STATUS_DRAFT)

    campaign = models.ForeignKey("Campaign", related_name="letters_rel")
    revision = models.IntegerField(_("Ревизия письма, увеличивается при редактировании"), default=0)
    code = models.CharField(_("Номер письма внутри рассылки"), max_length=16, default="A")
    description = models.TextField(_("Описание письма, для своих"), blank=True)
    template_engine = models.CharField(_("Язык темплейта"), max_length=32, null=True)

    # Заголовки письма
    subject = models.CharField(_("Тема письма"), max_length=1024)
    from_name = models.CharField(max_length=384)
    from_email = models.CharField(max_length=384)
    reply_to = models.CharField(max_length=128)
    charset = models.CharField(max_length=32, default="utf-8")

    # Содержание письма
    html_body = models.TextField()
    text_body = models.TextField()
    use_text_body = models.BooleanField(default=False)

    # Хеш от содержимого подгружаемого файла. Т.е. до разбора архива (если архив),
    #   до оборачивания ссылок, до вставки пикселя и т.п.
    original_letter_content_hash = models.CharField(max_length=64, null=True)

    # Разрешенные ссылки, чтобы закрыть openredirect. SENDER-196
    allowed_links = jsonfield.JSONField(null=True)
    allowed_domains = jsonfield.JSONField(null=True)

    # метаданные шаблона
    # variables_list - список переменных шаблона
    template_meta = jsonfield.JSONField(null=True)
    published = models.BooleanField(default=False)

    class Meta:
        verbose_name_plural = _("Письмо")

    def __str__(self):
        return "%s:%s" % (self.campaign, self.code)

    def drop_attachments(self):
        # Данные аттачей уезжают в S3, поэтому надо файлы чистить руками
        for att in self.attachments.all():
            att.delete()

    def get_attachment_by_uri(self, uri):
        return self.get_attachment(uri=uri)

    def get_attachment(self, **kw):
        try:
            return self.attachments.get(**kw)
        except ObjectDoesNotExist:
            return None

    def get_magic_list_id(self, full=False, testing=False):
        c = self.campaign
        list_id = c._get_magic_list_id()
        if c.is_ab:
            list_id = "%s-%s" % (list_id, self.code)

        if testing:
            list_id += "-testing"

        if full:
            return "%s.yandex.ru" % list_id
        else:
            return list_id

    @property
    def can_be_release_candidate(self):
        return not self.state == Letter.STATUS_DISABLED

    @property
    def uploaded(self):
        return bool(self.original_letter_content_hash)

    def update_template_meta(self, key, value):
        if self.template_meta is None:
            self.template_meta = {key: value}
        else:
            self.template_meta[key] = value

    def get_template_meta(self, key, default=None):
        d = self.template_meta
        if d is None:
            return default
        else:
            return d.get(key, default)

    def get_utm(self, use_default=True):
        if hasattr(self, "utm"):
            utm = self.utm
            return {
                "utm_source": utm.utm_source,
                "utm_medium": utm.utm_medium,
                "utm_campaign": utm.utm_campaign,
                "utm_content": utm.utm_content,
            }
        if use_default:
            return {
                "utm_source": "yandex",
                "utm_medium": "email",
                "utm_campaign": self.campaign.magic_list_id,
                "utm_content": self.code if self.campaign.type == self.campaign.TYPE_AB else None,
            }

    def get_template_variables(self):
        return self.get_template_meta("variables:user", default=())

    def add_allowed_domain(self, domain):
        domain = domain.lower()
        self.allowed_domains = (
            list(set(self.allowed_domains + [domain])) if self.allowed_domains else (domain,)
        )

    def verify_template_meta(self):
        verify_details = {}
        verify_result = True

        # Проверяем смогли ли распарсить шаблон
        parsed = self.get_template_meta("parsed", False)
        if not parsed:
            verify_result = False
            verify_details["template_error"] = self.get_template_meta("parse:error:message")
            verify_details["error_line_no"] = self.get_template_meta("parse:error:lineno")
            verify_details["result"] = verify_result
            return verify_details

        extensions_stat = self.get_template_meta("extensions:stats", default={})

        # Собираем разрешенные домены
        allowed_domains = set()
        if self.allowed_domains:
            allowed_domains.update(self.allowed_domains)

        for link in self.allowed_links or ():
            domain = urlparse(link)[1].lower()
            if domain:
                allowed_domains.add(domain)

        verify_details["allowed_domains"] = allowed_domains

        # Проверяем аттачи
        attachment_urls = list(self.attachments.all().values_list("uri", flat=True))
        used_attachments = extensions_stat.get("attachments", [])
        all_attachments = set(attachment_urls + used_attachments)
        attachment_details = {}
        for attach in all_attachments:
            used = attach in used_attachments
            uploaded = attach in attachment_urls
            error = used and not uploaded
            verify_result = verify_result and not error
            attachment_details[attach] = {
                "used": used,
                "uploaded": uploaded,
                "error": error,
            }

        verify_details["attachments"] = attachment_details
        verify_details["pixel_used"] = extensions_stat.get("pixel_usage", 0)

        if verify_details["pixel_used"] > 1:
            verify_result = False

        verify_details["result"] = verify_result

        return verify_details


def attachment_get_file_name(instance, filename):
    return ""


class LetterAttachment(TimestampsMixin, models.Model):
    """
    Вложения в письмо, например, картинки.
    """

    letter = models.ForeignKey("Letter", related_name="attachments")

    filename = models.CharField(max_length=1024, blank=True)
    file_size = models.IntegerField(default=0)
    uri = models.CharField(max_length=1024, null=True)
    mime_type = models.CharField(max_length=128, null=True)
    subtype = models.CharField(max_length=128, null=True)
    is_inline = models.BooleanField(default=False)

    source_url = models.CharField(max_length=1024, blank=True)
    publish_path = models.CharField(max_length=128, null=True, blank=True)

    class Meta:
        verbose_name_plural = _("Письмо:вложение")

    @property
    def content_type(self):
        return "%s/%s" % (self.mime_type, self.subtype)

    def unpublish(self, publisher):
        if self.publish_path:
            publisher.unpublish(self.publish_path)
        self.publish_path = None
        self.save(update_fields=["publish_path"])


class LetterUtm(models.Model):
    class Meta:
        verbose_name = "Письмо:Utm"
        verbose_name_plural = "Письмо:Utm"

    letter = models.OneToOneField(
        "Letter", related_name="utm", primary_key=True, on_delete=models.CASCADE
    )

    utm_source = models.CharField(max_length=256, blank=True)
    utm_medium = models.CharField(max_length=256, blank=True)
    utm_campaign = models.CharField(max_length=256, blank=True)
    utm_content = models.CharField(max_length=256, blank=True)
