from django.db import models
from django.utils.translation import ugettext_lazy as _
from jsonfield import JSONField
from fan.db.fields.slug import SlugDotField
from fan.utils.slug import create_slug
from .common import TimestampsMixin


class Maillist(TimestampsMixin, models.Model):
    account = models.ForeignKey("Account", related_name="maillists")
    slug = SlugDotField(unique=False, max_length=192, db_index=False)
    title = models.TextField(blank=True)
    filename = models.TextField(blank=True)
    data = models.BinaryField()
    preview = JSONField(default=list)
    size = models.IntegerField()

    class Meta:
        unique_together = (("account", "slug"),)

    def save(self, *args, **kwargs):
        if not self.slug:
            self.slug = create_slug()
        super().save(*args, **kwargs)


class SingleUseMailList(TimestampsMixin, models.Model):
    """
    Одноразовый список рассылки.
    Все адреса в одном csv-файле.
    """

    campaign = models.OneToOneField("Campaign", related_name="single_use_maillist")
    description = models.TextField(blank=True)
    data = models.BinaryField()  # csv-файл
    data_preview = JSONField(default=list)

    data_encoding = models.CharField(max_length=16, choices=(("gzip", "gzip"),), default="gzip")
    uploaded_at = models.DateTimeField(auto_now=True, null=True)
    uploaded_by = models.CharField(max_length=128)

    subscribers_number = models.IntegerField()
    subscribers_number_calculated_at = models.DateTimeField(null=True)

    # Хэш от содержимого, из которого получены данные.
    original_content_hash = models.CharField(max_length=64, null=True)

    class Meta:
        verbose_name_plural = _("Одноразовый список рассылки")
