from collections import defaultdict

from django.db import models
from django.db.models import Sum
from django.utils.translation import ugettext_lazy as _

from fan.delivery.status import DeliveryStatus

from .common import TimestampsMixin


class StatsRecord(TimestampsMixin, models.Model):
    """
    Абстрактная сводная запись статистики о письма/кампании.
    """

    class Meta:
        abstract = True

    stat_date = models.DateField(
        default=None, blank=True, null=True, verbose_name=_("Дата счетчика")
    )
    messages_processed = models.IntegerField(default=0, verbose_name=_("Обработано"))
    messages_delivered = models.IntegerField(default=0, verbose_name=_("Отправлено"))
    messages_not_delivered = models.IntegerField(default=0, verbose_name=_("Не отправлено"))
    control_group_count = models.IntegerField(
        default=0, verbose_name=_("Адресатов в контрольной группе")
    )

    reads = models.IntegerField(
        default=None, null=True, blank=True, verbose_name=_("Точно открыли")
    )
    opens = models.IntegerField(default=None, null=True, blank=True, verbose_name=_("Открытия"))

    clicks = models.IntegerField(default=None, null=True, blank=True, verbose_name=_("Клики"))
    clicks_unique = models.IntegerField(
        default=None, null=True, blank=True, verbose_name=_("Кликнувшие")
    )
    deletes = models.IntegerField(default=None, null=True, blank=True, verbose_name=_("Удаления"))
    spam = models.IntegerField(default=None, null=True, blank=True, verbose_name=_("Спам"))
    unsubscribes = models.IntegerField(
        default=None, null=True, blank=True, verbose_name=_("Отписались")
    )
    unsubscribes_global = models.IntegerField(
        default=None, null=True, blank=True, verbose_name=_("Отписались")
    )

    bounces = models.IntegerField(
        default=None, null=True, blank=True, verbose_name=_("Писем вернулось")
    )
    bounces_permanent = models.IntegerField(default=None, null=True, blank=True)
    bounces_temporary = models.IntegerField(default=None, null=True, blank=True)


class StatsCampaign(StatsRecord):
    """
    Статистика кампании
    """

    campaign = models.OneToOneField("fan.Campaign", related_name="stats_rel")

    class Meta:
        verbose_name_plural = _("Статистика по кампании")

    def __str__(self):
        return "campaign stats for {}".format(self.campaign)


class DeliveryErrorStats(models.Model):
    """
    Счётчики ошибок отправки
    """

    campaign = models.ForeignKey(
        "fan.Campaign", related_name="deliveryerrorstats_set", db_constraint=False
    )
    letter = models.ForeignKey(
        "fan.Letter", related_name="deliveryerrorstats_set", db_constraint=False, null=True
    )
    status = models.IntegerField(choices=DeliveryStatus.CHOICES, null=True)
    stat_date = models.DateField(default=None, blank=True, null=True)
    count = models.IntegerField(default=0)

    class Meta:
        unique_together = (("stat_date", "campaign", "letter", "status"),)
