from django.db import models
from django.utils.translation import ugettext_lazy as _

from jsonfield import JSONField

from fan.utils.emails import clean

from .common import TimestampsMixin
from .editoriallog import EditorialLog
import collections


class AccountUnsubscribeListManager(models.Manager):
    def create_default(self, account):
        from fan.utils.slug import create_slug

        return self.get_or_create(
            account=account,
            default=True,
            defaults={
                "name": {"ru": AccountUnsubscribeList.DEFAULT_NAME},
                "slug": create_slug(),
                "visible": False,
            },
        )

    def create_general(self, account):
        from fan.utils.slug import create_slug

        return self.get_or_create(
            account=account,
            general=True,
            defaults={
                "name": {"ru": AccountUnsubscribeList.GENERAL_NAME},
                "slug": create_slug(),
                "visible": False,
            },
        )

    def get_default(self, account):
        return self.get(account=account, default=True)

    def get_general(self, account):
        return self.get(account=account, general=True)

    def get_recipient_data(self, email, account, visible):
        unsub_lists = self.filter(account=account, visible=visible)
        result = {"lists": [], "group": None}

        try:
            email = clean(email)
            for lst in unsub_lists:
                result["lists"].append(
                    dict(list=lst, unsubscribed=lst.elements.filter(email=email).exists())
                )

            group = account.unsub_domains.first()
            if group:
                result["group"] = {
                    "object": group,
                    "unsubscribed": group.elements.filter(email=email).exists(),
                }

        except ValueError:
            pass

        return result

    def modified_since(self, dt):
        return self.filter(elements__created_at__gte=dt).distinct()


class AccountUnsubscribeList(models.Model):
    """
    Список отписки аккаунта
    """

    DEFAULT_NAME = "Основной"
    GENERAL_NAME = "Глобальный"

    name = JSONField(verbose_name="Название")
    account = models.ForeignKey(
        "fan.Account", verbose_name="Аккаунт", related_name="unsubscribe_lists"
    )
    slug = models.SlugField(unique=False, max_length=192, db_index=False, null=False)
    visible = models.BooleanField(default=False, verbose_name="Отдавать список во внешнее API")
    description = JSONField(verbose_name="Описание", null=True, blank=True)
    counter = models.PositiveIntegerField(verbose_name="Количество отписок", default=0)

    created_by = models.CharField(max_length=64, editable=False, null=True, blank=True)
    default = models.BooleanField(verbose_name="Список по-умолчанию", default=False)
    general = models.BooleanField(verbose_name="Глобальный список аккаунта", default=False)

    objects = AccountUnsubscribeListManager()

    def save(self, *args, **kwargs):
        from fan.utils.slug import create_slug

        if not self.slug:
            self.slug = create_slug()

        super().save(*args, **kwargs)

    class Meta:
        verbose_name = _("Список отписки")
        verbose_name_plural = _("Списки отписки")

    def __str__(self):
        return "{0}: {1}".format(self.account, self.get_name())

    def update_counter(self):
        """
        Обновляем счетчик отписок
        """

        self.counter = self.elements.count()
        self.save()

    def upsert_element(self, email):
        """
        Добавить email в список
        :param email:
        """
        _email = clean(email)
        return UnsubscribeListElement.objects.get_or_create(
            list=self, email__iexact=_email, defaults={"email": _email}
        )

    def bulk_upsert_elements(self, iterable, update_counter=False, on_process_element=None):
        """
        Добавить email'ы к списку отписки

        :param iterable: список email'ов
        :param update_counter: обновлять ли количество отписок
        :param on_process_element: колбек после вставки очередного елемента
        """

        for email in iterable:
            try:
                _, created = self.upsert_element(email)
                if created and isinstance(on_process_element, collections.Callable):
                    on_process_element(email=email, unsub_list=self)
            except ValueError:
                continue

        if update_counter:
            self.update_counter()

    def editorial_logs(self):
        return EditorialLog.objects.filter(
            object_type="unsubscribe_list", object_id=self.id
        ).order_by("-datetime")

    def remove(self, email):
        """
        Удалить из списка
        """
        try:
            self.elements.filter(email__iexact=clean(email)).delete()
        except ValueError:
            pass

    def get_name(self, locale="ru"):
        if isinstance(self.name, dict):
            return self.name.get(locale)

        return self.name

    def get_description(self, locale="ru"):
        if isinstance(self.description, dict):
            return self.description.get(locale) if self.description else None

        return self.description


class UnsubscribeListElement(TimestampsMixin, models.Model):
    """
    Отписка пользователя
    """

    list = models.ForeignKey(
        "fan.AccountUnsubscribeList", verbose_name="Список отписки", related_name="elements"
    )
    email = models.CharField(max_length=255, db_index=True)

    class Meta:
        verbose_name = _("Список отписки: отписка")
        verbose_name_plural = _("Списки отписки: отписки")
