import string


def generate_converter(alphabet):
    base_ = len(alphabet)
    reverse_alphabet = dict([(char, pos) for pos, char in enumerate(alphabet)])

    def _convert(v):

        if v == 0:
            yield alphabet[0]
            raise StopIteration

        if v < 0:
            v = abs(v)
            yield "-"

        while v:
            v, rem = divmod(v, base_)
            yield alphabet[rem]

    def encode(n):
        return "".join([c for c in _convert(n)])

    def decode(s):

        if s[0] == "-":
            negative = True
            s = s[1:]
        else:
            negative = False

        num, m = 0, 1
        for char in s:
            num += reverse_alphabet[char] * m
            m = m * base_
        if negative:
            return -num
        else:
            return num

    return encode, decode


to_base26, from_base26 = generate_converter(string.ascii_uppercase)
to_base36, from_base36 = generate_converter(string.digits + string.ascii_uppercase)


def test_base_converter():
    assert to_base26(0) == "A"
    assert to_base26(1) == "B"
    assert to_base26(-1) == "-B"

    for val in [-5300, 2 ** 53, 2 ** 53 + 1] + list(range(65536, 65536 * 3)):
        try:
            assert from_base26(to_base26(val)) == val
            assert from_base36(to_base36(val)) == val
        except ValueError:
            print(val)


if __name__ == "__main__":
    import sys

    for title, value in [("sys.maxint", sys.maxsize), ("2**53", 2 ** 53)]:
        for func in (to_base26, to_base36):
            print(title, func.__name__, value, func(value))
    test_base_converter()
