import collections
import functools
from time import time

from django.core.cache import cache

from cachetools.func import RLock, _marker, _NLock, hashkey, typedkey
from cachetools.lru import LRUCache

default_cache = cache


_CacheInfo = collections.namedtuple(
    "CacheInfo", ["hits", "misses", "stales", "maxsize", "currsize"]
)


class TimedLRUCache(LRUCache):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.__time = {}

    def __getitem__(self, key):
        value = super().__getitem__(key)
        _time = self.__time.get(key, None)
        return value, _time

    def __setitem__(self, key, value):
        super().__setitem__(key, value)
        self.__time[key] = time()

    def __delitem__(self, key):
        super().__delitem__(key)
        self.__time.pop(key, None)


def _ttl_cache(cache, ttl, typed=False, context=_marker, use_stale_on=None):
    def decorator(func):
        key = typedkey if typed else hashkey
        if context is _marker:
            lock = RLock()
        elif context is None:
            lock = _NLock()
        else:
            lock = context()
        stats = [0, 0, 0]

        def cache_info():
            with lock:
                hits, misses, stales = stats
                maxsize = cache.maxsize
                currsize = cache.currsize
            return _CacheInfo(hits, misses, stales, maxsize, currsize)

        def cache_clear():
            with lock:
                try:
                    cache.clear()
                finally:
                    stats[:] = [0, 0, 0]

        def wrapper(*args, **kwargs):
            k = key(*args, **kwargs)
            with lock:
                expired = False
                now = time()
                v = _marker
                try:
                    v, _time = cache[k]
                    expired = (now - _time) > ttl
                    if not expired:
                        stats[0] += 1
                        return v
                except KeyError:
                    pass

            try:
                stats[1] += 1
                v = func(*args, **kwargs)
            except Exception as exc:
                if use_stale_on and isinstance(exc, use_stale_on):
                    if expired and v is not _marker:
                        stats[2] += 1
                        return v
                raise

            try:
                with lock:
                    cache[k] = v
            except ValueError:
                pass  # value too large
            return v

        functools.update_wrapper(wrapper, func)
        if not hasattr(wrapper, "__wrapped__"):
            wrapper.__wrapped__ = func  # Python < 3.2
        wrapper.cache_info = cache_info
        wrapper.cache_clear = cache_clear
        return wrapper

    return decorator


def ttl_cache(maxsize=128, ttl=600, getsizeof=None, use_stale_on=None):
    """Decorator to wrap a function with a memoizing callable that saves
    up to `maxsize` results based on a Least Recently Used (LRU)
    algorithm, expiring items after `ttl` seconds.
    If `use_stale_on` is non-empty list of exceptions and func raises exception from this list,
    ignores expiration and return from cache.
    """
    return _ttl_cache(TimedLRUCache(maxsize, getsizeof), ttl=ttl, use_stale_on=use_stale_on)
