from django.conf import settings

from fan.exceptions import DATABASE_EXCEPTIONS
from fan.accounts.get import (
    get_account_by_name as get_account_by_name_impl,
    get_accounts_by_user_id as get_accounts_by_user_id_impl,
)
from fan.campaigns.get import get_campaign_by_slug as get_campaign_by_slug_impl
from fan.campaigns.list import get_campaigns as get_campaigns_impl
from fan.models import (
    AccountUnsubscribeList,
    Campaign,
    Letter,
)

from fan.utils.cache import ttl_cache

CACHE_PARAMS = dict(maxsize=1024, ttl=120, use_stale_on=(DATABASE_EXCEPTIONS,))


@ttl_cache(**CACHE_PARAMS)
def get_campaign(id, account_id=None, raise_if_needed=True, skip_deleted=True):
    if skip_deleted:
        qs = Campaign.objects.exclude(deleted=True)
    else:
        qs = Campaign.objects

    if account_id:
        qs = qs.filter(account_id=account_id)

    try:
        return qs.get(id=id)
    except Campaign.DoesNotExist:
        if raise_if_needed:
            raise


@ttl_cache(**CACHE_PARAMS)
def get_campaigns(account_id, states):
    return get_campaigns_impl(account_id, states)


@ttl_cache(**CACHE_PARAMS)
def get_accounts_by_user_id(user_id):
    return get_accounts_by_user_id_impl(user_id)


@ttl_cache(**CACHE_PARAMS)
def get_account_by_name(name):
    return get_account_by_name_impl(name)


@ttl_cache(**CACHE_PARAMS)
def get_campaign_by_slug(slug, account_id):
    return get_campaign_by_slug_impl(slug, account_id)


@ttl_cache(**CACHE_PARAMS)
def get_letter(id, campaign_id=None, raise_if_needed=True):
    try:
        letter = Letter.objects.get(id=id)
        if campaign_id and letter.campaign_id != campaign_id:
            raise Letter.DoesNotExist()
        return letter
    except Letter.DoesNotExist:
        if raise_if_needed:
            raise


@ttl_cache(**CACHE_PARAMS)
def get_campaign_unsubscribe_list_exists(campaign_id):
    """
    Проверка на существование списков отписки
    """
    return AccountUnsubscribeList.objects.filter(campaigns__id=campaign_id).exists()


@ttl_cache(**CACHE_PARAMS)
def get_campaign_unsubscribe_lists_id(campaign_id):
    """
    Получение идентификаторов всех списков отписки для кампании
    """
    campaign = get_campaign(campaign_id, raise_if_needed=False)
    if campaign:
        lists_ids = list(
            AccountUnsubscribeList.objects.filter(campaigns__id=campaign.id).values_list(
                "id", flat=True
            )
        ) + list(
            AccountUnsubscribeList.objects.filter(
                account_id=campaign.account.id, general=True
            ).values_list("id", flat=True)
        )
        return lists_ids
    return []


all_caches = (
    get_account_by_name,
    get_accounts_by_user_id,
    get_campaign,
    get_campaigns,
    get_campaign_by_slug,
    get_campaign_unsubscribe_list_exists,
    get_campaign_unsubscribe_lists_id,
    get_letter,
)


def cache_clear():
    for c in all_caches:
        c.cache_clear()
