import re

from .email_normalizer import get_normalizer

# fixed
_FORMAT_CHECK = re.compile(
    r"^[абвгдеёжзийклмнопрстуфхцчшщъыьэюяa-z0-9.!#$%&'*+\/=?^_`{|}~-]{1,64}@[абвгдеёжзийклмнопрстуфхцчшщъыьэюя\w\d](?:[абвгдеёжзийклмнопрстуфхцчшщъыьэюя\w\d-]{0,61}[абвгдеёжзийклмнопрстуфхцчшщъыьэюя\w\d])?(?:\.[абвгдеёжзийклмнопрстуфхцчшщъыьэюя\w\d](?:[абвгдеёжзийклмнопрстуфхцчшщъыьэюя\w\d-]{0,61}[абвгдеёжзийклмнопрстуфхцчшщъыьэюя\w\d])?)*$",
    re.U,
)


def is_valid(email):
    return bool(_FORMAT_CHECK.match(email.lower()))


def is_valid_localpart(localpart):
    return is_valid(localpart + "@domain.ru")


def get_local_part(email):
    local_part, _, _ = email.partition("@")
    return local_part


def get_domain(email):
    _, at, domain = email.lower().rpartition("@")
    if at:
        return domain
    return ""


def normalize(email):
    if not is_valid(email):
        raise ValueError("Wrong email format")
    local_part = get_local_part(email).lower()
    domain = get_domain(email)
    return get_normalizer(domain).normalize(local_part, domain)


def clean(email, raise_on_error=True):
    """
    Отчищаем и нормализуем email
    """
    try:
        return normalize(email.strip())
    except (UnicodeDecodeError, ValueError):
        if raise_on_error:
            raise ValueError("Wrong email format")
