import functools
import logging
import sys
from datetime import datetime
from django.utils import timezone
from contextlib import contextmanager
from logging import Filter
from multiprocessing import Lock
from fan.utils import json
from tskv_logging.formatters import TSKVFormatter as TSKVFormatterBase


DEFAULT_EXCLUDED_FIELDS = set(
    (
        "args",
        "created",
        "filename",
        "lineno",
        "module",
        "msecs",
        "pathname",
        "processName",
        "relativeCreated",
        "msg",
        "thread",
        "full_timestamp",
        "unixtime",
        "pid",
        "process",
    )
)

DEFAULT_ORDERED_FIRST = (
    "tskv_format",
    "timestamp",
    "timezone",
)

DEFAULT_RENAME_FIELDS = {
    "timestamp_hack": "timestamp",
    "timezone_hack": "timezone",
}


class LockedHandler:
    """Wraps logging handler with multiprocessing.Lock.

    Used to synchronize log writes between processes with common ancestor, e. g. celery worker pool.
    The common ancestor process must create handler instance before children, so they can share handler.

    All writes via handler.emit() calls will be serialized.

    One main process should create at most one file handler per log file and wrap it in locked handler.

    This is not fully compatible Handler class.
    In case one wants to extend logging.FileHandler class to have fully compatible handler,
    he should prefer to redefine getLock method to return multiprocessing.RLock instance,
    because default logging.FileHandler uses threading.RLock.

    """

    def __init__(self, log_handler=None):
        """Inits wrapper.
        Args:
            log_handler - handler from logging.handlers
        """
        self._log_handler = log_handler
        self._lock = Lock()

    def emit(self, record):
        with self._lock:
            return self._log_handler.emit(record)


class TSKVFormatter(TSKVFormatterBase):
    def format(self, record):
        msg = super().format(record).decode("utf-8")
        return msg.strip()


@contextmanager
def duplicate_log_to_stream(level="INFO", stream=sys.stdout, dry_run=False):
    """
    Handful context manager to duplicate log output to stdout (by default), e. g. used in manage.py commands.
    """
    root_logger = logging.getLogger()
    already_in_context = getattr(root_logger, "_log_with_stdout", False)
    if dry_run or already_in_context:
        yield
    else:
        stream_handler = logging.StreamHandler(stream)
        stream_handler.setLevel(level)
        root_logger.addHandler(stream_handler)
        root_logger._log_with_stdout = True
        try:
            yield
        finally:
            root_logger.removeHandler(stream_handler)
            del root_logger._log_with_stdout


def log_function_call(name=None):
    """Декоратор, логирующий вход и выход из функции для менеджмент-комманд"""

    def decorator(func):

        function_name = name or func.__name__

        @functools.wraps(func)
        def wrapped(*args, **kwargs):
            logging.info("Call function %s" % function_name)
            try:
                result = func(*args, **kwargs)
                logging.info("Done function %s" % function_name)
                return result
            except Exception:
                logging.info("Error in function %s" % function_name)
                raise

        return wrapped

    return decorator


def encode_complex_or_none(data):
    """
    Списки и словари записать JSON-ом.
    None - как пустые строки.
    """
    for k, v in data.items():
        if _is_complex(v):
            data[k] = json.dumps(v)
        if v is None:
            data[k] = ""
    return data


# hack to avoid changing substitution of these field in contrib
def inject_timestamp_hack(fields):
    tz = timezone.get_default_timezone()
    now = datetime.now(tz=tz)
    fields.update(
        {
            "timestamp_hack": now.strftime("%Y-%m-%d %H:%M:%S.%f"),
            "timezone_hack": now.strftime("%z"),
        }
    )
    return fields


def _is_complex(value):
    return isinstance(value, (dict, list))
