def get_user_ip_from_meta(meta):
    real_ip = meta.get("HTTP_X_FORWARDED_FOR") or meta.get("HTTP_X_REAL_IP")
    if real_ip:
        ip = real_ip.split(",")[0]
    else:
        ip = meta.get("REMOTE_ADDR")
    return ip


def get_user_ip(request):
    return get_user_ip_from_meta(request.META)


def get_user_agent(request):
    return request.META.get("HTTP_USER_AGENT", None)


def get_host(request):
    return request.META.get("HTTP_HOST", None)


def iter_safe_cookies(cookies):
    """
    Удалим  куки, которые могут содержать данные о сессии.
    """
    for k, v in list(cookies.items()):
        if "session" in k.lower():
            continue
        yield k, v


COMMON_METADATA = {"userip", "cookies", "useragent", "host"}


def get_request_metadata(request, fields=COMMON_METADATA):
    """
    Возвращаем метаданные запроса:
    - ip-адрес пользователя
    - user-agent
    - куки
    :param request:
    :return:
    """
    result = {}
    if request:
        if "userip" in fields:
            result["userip"] = get_user_ip(request)
        if "cookies" in fields:
            result["cookies"] = dict(iter_safe_cookies(request.COOKIES))
        if "useragent" in fields:
            result["useragent"] = get_user_agent(request)
        if "host" in fields:
            result["host"] = get_host(request)
    return result


class RequestData(dict):

    PRESERVE_METADATA = COMMON_METADATA

    @classmethod
    def from_request(cls, request):
        return cls(get_request_metadata(request, fields=cls.PRESERVE_METADATA))
