import os
from logging import getLogger

from django.conf import settings

import cachetools
import requests
from ticket_parser2_py3.api.v1 import BlackboxClientId
from tvm2 import TVM2Qloud

from utvm import TVM, TvmException
from utvm.utvm import TVM_KEYS_REFRESH_INTERVAL, TicketCheckResult

logger = getLogger(__name__)

TVMError = TvmException

if (
    os.environ.get("QLOUD_TVM_TOKEN") is not None
    and os.environ.get("ENABLE_TVM_DAEMON", "flase").lower() == "true"
):
    logger.debug("active tvm client id: ", settings.TVM_CLIENT)
    TVM2_DAEMON = TVM2Qloud(settings.TVM_CLIENT, BlackboxClientId.Prod, allowed_clients={"*"})
else:
    # logger.warn('QLOUD_TVM_TOKEN is not set, tvm daemon will be unavailable')
    TVM2_DAEMON = None

TVM_CFG = TVM(
    settings.TVM_CLIENT,
    settings.TVM_SECRET,
    use_v1=False,
    update_keys_interval=settings.TVM_UPDATE_PUBLIC_KEYS_INTERVAL,
)

if settings.TVM_USE_TEST_TICKET:
    TVM_CFG.force_tvm2_ticket("", settings.TEST_TVM_TICKET)


class TVMWithDaemon(TVM):

    """
    Обертка над utvm.TVM, которая фолбечиться на tvm демона
    при проверке сервис-тикетов. Нужно как минимум при дебаге
    для истользования тикетов из tvmtool. В перспективе хочется
    полностью перейти на демона
    """

    def __init__(
        self,
        client_id,
        secret,
        tvm_host="tvm-api.yandex.net",
        local_v1_keys=None,
        allowed_v1_clients=None,
        passport_env=None,
        use_v1=False,
        update_keys_interval=TVM_KEYS_REFRESH_INTERVAL,
        tvm_daemon=None,
    ):
        super().__init__(
            client_id,
            secret,
            tvm_host,
            local_v1_keys,
            allowed_v1_clients,
            passport_env,
            use_v1,
            update_keys_interval,
        )
        self._tvm_daemon = tvm_daemon
        self._checker = self.ticket_checker()
        if settings.TVM_USE_UNITTEST_PUBLIC_KEYS:
            self.force_tvm2_keys(settings.TVM_UNITTEST_PUBLIC_KEYS)

    def check_service_ticket(self, service_ticket):
        check_result = self._checker.check_tickets(tvm_2_service_ticket=service_ticket)
        if check_result is None or check_result.src is None:
            check_result = self._check_service_ticket_via_daemon(service_ticket) or check_result
        return check_result

    def _check_service_ticket_via_daemon(self, service_ticket):
        if self._tvm_daemon is None:
            return None
        parse_result = self._tvm_daemon.parse_service_ticket(ticket=service_ticket)
        if parse_result is not None and parse_result.src is not None:
            return TicketCheckResult(parse_result, None)
        return None


def sessions_producer(tvm):

    cache = cachetools.TTLCache(128, 1800)
    ticket_getter = tvm.ticket_getter()

    @cachetools.cached(cache=cache)
    def get_ticket_headers(dst, scopes):
        return ticket_getter.get_service_ticket_headers(dst, scopes=scopes)[str(dst)]

    class TvmSession(requests.Session):
        def __init__(self, tvm_dst, scopes=None, *args, **kwargs):
            super().__init__(*args, **kwargs)
            if not isinstance(self.verify, str):
                self.verify = "/etc/ssl/certs/ca-certificates.crt"
            self._dst = tvm_dst
            self._scopes = scopes

        def request(self, *args, **kwargs):
            headers = kwargs.pop("headers", None)
            tvm_headers = get_ticket_headers(self._dst, self._scopes)
            if headers:
                for k, v in list(tvm_headers.items()):
                    headers[k] = v
            else:
                headers = tvm_headers

            return super().request(*args, headers=headers, **kwargs)

    return TvmSession


def ticket_producer(tvm):
    ticket_cache = cachetools.TTLCache(128, 5)
    ticket_getter = tvm.ticket_getter()

    @cachetools.cached(cache=ticket_cache)
    def get_ticket_headers(dst, scopes):
        return ticket_getter.get_service_ticket_headers(dst, scopes=scopes)[str(dst)]

    class TvmTicketProducer:
        def __init__(self, tvm_dst, scopes=None):
            self._dst = int(tvm_dst)
            self._scopes = scopes

        def get_ticket(self):
            tvm_headers = get_ticket_headers(self._dst, self._scopes)
            return tvm_headers["X-Ya-Service-Ticket"]

    return TvmTicketProducer


TvmSession = sessions_producer(TVM_CFG)
TvmTicketProducer = ticket_producer(TVM_CFG)

TVM_CFG_WITH_DAEMON = TVMWithDaemon(
    settings.TVM_CLIENT,
    settings.TVM_SECRET,
    use_v1=False,
    update_keys_interval=settings.TVM_UPDATE_PUBLIC_KEYS_INTERVAL,
    tvm_daemon=TVM2_DAEMON,
)
