import csv
from io import StringIO

import xlrd

# Based on code http://code.activestate.com/recipes/546518
# by Philip (flip) Kromer flip@infochimp.org
# (released under the Python license)


def tupledate_to_isodate(tupledate):
    """
    Turns a gregorian (year, month, day, hour, minute, nearest_second) into a
    standard YYYY-MM-DDTHH:MM:SS ISO date.  If the date part is all zeros, it's
    assumed to be a time; if the time part is all zeros it's assumed to be a date;
    if all of it is zeros it's taken to be a time, specifically 00:00:00 (midnight).

    Note that datetimes of midnight will come back as date-only strings.  A date
    of month=0 and day=0 is meaningless, so that part of the coercion is safe.
    For more on the hairy nature of Excel date/times see http://www.lexicon.net/sjmachin/xlrd.html
    """
    (y, m, d, hh, mm, ss) = tupledate

    def nonzero(n):
        return n != 0

    date = "%04d-%02d-%02d" % (y, m, d) if list(filter(nonzero, (y, m, d))) else ""
    time = (
        "T%02d:%02d:%02d" % (hh, mm, ss) if list(filter(nonzero, (hh, mm, ss))) or not date else ""
    )
    return date + time


def format_excelval(book, type, value, wanttupledate):
    """Clean up the incoming excel data"""
    # Data Type Codes:
    # EMPTY   0
    # TEXT    1 a Unicode string
    # NUMBER  2 float
    # DATE    3 float
    # BOOLEAN 4 int; 1 means TRUE, 0 means FALSE
    # ERROR   5
    if type == 2:  # TEXT
        if value == int(value):
            value = int(value)
    elif type == 3:  # NUMBER
        datetuple = xlrd.xldate_as_tuple(value, book.datemode)
        value = datetuple if wanttupledate else tupledate_to_isodate(datetuple)
    elif type == 5:  # ERROR
        value = xlrd.error_text_from_code[value]
    return value


class XLSFile:
    def __init__(self, data):
        if hasattr(data, "read"):
            data = data.read()
        self.book = xlrd.open_workbook(file_contents=data)
        self.sheet_names = self.book.sheet_names()

    def sheet_as_array(self, sheet_name):
        """
        Returns a list of sheets; each sheet is a dict containing
        * sheet_name: unicode string naming that sheet
        * sheet_data: 2-D table holding the converted cells of that sheet
        """

        def formatter(pair):
            t, v = pair
            return format_excelval(self.book, t, v, False)

        if isinstance(sheet_name, int):
            raw_sheet = self.book.sheet_by_index(sheet_name)
        else:
            raw_sheet = self.book.sheet_by_name(sheet_name)
        data = []
        for row in range(raw_sheet.nrows):
            (types, values) = (raw_sheet.row_types(row), raw_sheet.row_values(row))
            data.append(list(map(formatter, list(zip(types, values)))))
        return data

    def __getitem__(self, item):
        return self.sheet_as_array(item)

    def as_array(self):
        """
        Returns a list of sheets; each sheet is a list containing
        * sheet_name: unicode string naming that sheet
        * sheet_data: 2-D table holding the converted cells of that sheet
        """
        for sheet_name in self.sheet_names:
            yield sheet_name, self.sheet_as_array(sheet_name)

    def sheet_as_csv_stream(self, sheet_name):
        """Превращаем содержимое xls[x] в поток CSV символов.

        double-quote CSV корректно обрабатываются существующим кодом при загрузке CSV
        поэтому его же мы будем использовать при конвертации xlsx -> csv

        поскольку quoting по-умолчанию включен, необходимости в escapechar нет -
        соответствующая строка будет обрамлена символами '"', а ее собственные символы '"'
        будут предварены дополнительным '"'

        Наконец, сконвертированные из xlsx с escape csv-шки все равно читались некорректно.

        """

        stream = StringIO()
        csvout = csv.writer(stream, delimiter=",", doublequote=True)
        csvout.writerows(list(self.sheet_as_array(sheet_name)))
        stream.seek(0)
        return stream

    def sheet_as_csv(self, sheet_name):
        stream = self.sheet_as_csv_stream(sheet_name)
        return stream.read()


def xls_to_csv(data, sheet_index=0):
    return XLSFile(data).sheet_as_csv(sheet_name=sheet_index)
