import os.path

from library.python import resource

from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import BaseComponent
from mail.devpack.lib.components.fakebb import FakeBlackbox
from mail.devpack.lib.components.pyremock import PyremockComponent
from mail.devpack.lib.yhttp_service import YplatformHttpService
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.tvmapi import TvmApi

from mail.pg.furitadb.devpack.components.furitadb import FuritaDb


class FuritaBase(BaseComponent):
    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(FuritaBase, cls).gen_config(port_generator, config=config)
        http = YplatformHttpService.gen_config(port_generator, config=config)
        return {
            **base,
            **http,
        }

    def __init__(self, env, components):
        super().__init__(env, components)
        self.yhttp = YplatformHttpService(env, self.NAME, binary_name='furita', custom_path='furita/src')

    def init_root(self):
        self.yhttp.init_root()

        self.write_resource('furita/base.yml', os.path.join(self.etc_path, 'base.yml'))
        self.write_resource('furita/query_pq.conf', os.path.join(self.etc_path, 'query_pq.conf'))
        self.write_resource('furita/furitadb.conf', os.path.join(self.etc_path, 'furitadb.conf'))
        self.write_resource('furita/tvm_secret', os.path.join(self.etc_path, 'tvm_secret'))

        self.write_resource('furita/filter_confirm_forward_en.eml', os.path.join(self.etc_path, 'filter_confirm_forward_en.eml'))
        self.write_resource('furita/filter_confirm_forward_ru.eml', os.path.join(self.etc_path, 'filter_confirm_forward_ru.eml'))
        self.write_resource('furita/filter_confirm_notify_en.eml', os.path.join(self.etc_path, 'filter_confirm_notify_en.eml'))
        self.write_resource('furita/filter_confirm_notify_ru.eml', os.path.join(self.etc_path, 'filter_confirm_notify_ru.eml'))

        self._init_root()

    def _init_root(self):
        pass

    def write_resource(self, resource_path, destination_path):
        helpers.write2file(what=resource.find(resource_path), path=destination_path)

    @property
    def etc_path(self):
        return self.yhttp.get_etc_path()

    def get_root(self):
        return self.yhttp.get_root()

    def webserver_port(self):
        return self.yhttp.webserver_port

    def start(self):
        self.yhttp.start('pong')

    def stop(self):
        self.yhttp.stop()

    def restart(self):
        self.stop()
        self.start()

    def info(self):
        info = self.yhttp.info()
        return {
            **info,
            'state': self.state
        }

    def ping(self):
        return self.yhttp.ping()


class FuritaComponent(FuritaBase):
    NAME = 'furita'
    DEPS = [Sharpei, TvmApi, FuritaDb, FakeBlackbox, PyremockComponent]

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(FuritaComponent, cls).gen_config(port_generator, config=config)
        return {
            **base,
            'msearch_port': next(port_generator),
            'mops_port': next(port_generator),
            'so_check_form_port': next(port_generator),
        }

    def make_config(self, params={}):
        default_params = dict(
            tvm_api_port=self.components[TvmApi].port,
            sharpei_port=self.components[Sharpei].webserver_port(),
            msearch_port=self._msearch_port,
            mops_port=self._mops_port,
            so_check_form_port=self._so_check_form_port,
            furitadb_port=self.components[FuritaDb].port(),
            blackbox_port=self.components[FakeBlackbox].port,
            pyremock_port=self.components[PyremockComponent].port,
            allow_forward='none',
        )
        params = {**default_params, **params}
        return self.yhttp.format_config(
            config=resource.find('furita/devpack.yml'),
            **params
        )

    def __init__(self, env, components):
        super().__init__(env, components)
        self._msearch_port = self.config[self.name]['msearch_port']
        self._mops_port = self.config[self.name]['mops_port']
        self._so_check_form_port = self.config[self.name]['so_check_form_port']

    def _init_root(self):
        devpack_config = self.make_config()
        helpers.write2file(what=devpack_config, path=self.yhttp.get_config_path())

    @property
    def msearch_port(self):
        return self._msearch_port

    @property
    def mops_port(self):
        return self._mops_port

    @property
    def so_check_form_port(self):
        return self._so_check_form_port

    def get(self, target, timeout=10, params=None, headers={}):
        return self.yhttp.get(target=target, timeout=timeout, params=params, headers=headers)


class FuritaTesting(FuritaBase):
    NAME = 'furita_testing'
    DEPS = []

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(FuritaTesting, cls).gen_config(port_generator, config=config)
        return base

    def __init__(self, env, components):
        super().__init__(env, components)

    def _init_root(self):
        devpack_config = self.yhttp.format_config(config=resource.find('furita/testing.yml'))
        helpers.write2file(what=devpack_config, path=self.yhttp.get_config_path())

        self.yhttp.init_pgpass(self.get_root())

    def start(self):
        self.yhttp.put_pgpassfile_in_env(self.get_root())
        super().start()
