#pragma once

#include <string>
#include <list>
#include <set>
#include <vector>
#include <ctime>
#include <boost/shared_ptr.hpp>
#include <pgg/enumeration.h>

namespace furita::rules {

struct condition {
    struct _LinkType {
        enum Enum {
            OR = 0,
            AND
        };
        template <typename Map>
        void fill(Map& m) const {
            m.insert(typename Map::value_type(OR, "or"));
            m.insert(typename Map::value_type(AND, "and"));
        }
        using Filler = _LinkType;
    };
    using link_type = pgg::Enumeration<_LinkType, _LinkType::OR>;

    struct _OperType {
        enum Enum {
            MATCHES = 0,
            CONTAINS,
            EXISTS
        };
        template <typename Map>
        void fill(Map& m) const {
            m.insert(typename Map::value_type(MATCHES, "matches"));
            m.insert(typename Map::value_type(CONTAINS, "contains"));
            m.insert(typename Map::value_type(EXISTS, "exists"));
        }
        using Filler = _OperType;
    };
    using oper_type = pgg::Enumeration<_OperType, _OperType::CONTAINS>;

    condition()
        : link(link_type::OR)
        , oper(oper_type::CONTAINS)
        , neg(false)
    {
    }

    condition(const std::string& field, const std::string& pattern,
              link_type link, oper_type oper, bool neg)
        : field(field)
        , pattern(pattern)
        , link(link)
        , oper(oper)
        , neg(neg)
    {
    }

    uint64_t rule_id;
    std::string field, field_type, pattern;
    link_type link;
    oper_type oper;
    bool neg;
};

using condition_ptr = boost::shared_ptr<condition>;
using condition_list = std::vector<condition_ptr>;
using condition_list_ptr = boost::shared_ptr<condition_list>;

struct action {
    action()
        : verified(false)
    {
    }

    action(const std::string& oper, const std::string& param)
        : oper(oper)
        , param(param)
        , verified(true)
    {
    }

    uint64_t rule_id, id;
    std::string oper, param;
    bool verified;
};

using action_ptr = boost::shared_ptr<action>;
using action_list = std::vector<action_ptr>;
using action_list_ptr = boost::shared_ptr<action_list>;

struct rule {
    rule()
        : prio(0)
        , enabled(true)
        , stop(false)
        , created(0)
        , type("user")
        , conditions(new condition_list)
        , actions(new action_list)
    {
    }

    std::string name;
    int64_t prio;
    bool enabled, stop;
    std::time_t created;
    std::string type;
    condition_list_ptr conditions;
    action_list_ptr actions;
    uint64_t id, uid;
};

using rule_ptr = boost::shared_ptr<rule>;
using rule_list = std::vector<rule_ptr>;
using rule_list_ptr = boost::shared_ptr<rule_list>;

struct blacklist_entry {
    blacklist_entry()
        : created(0)
    {
    }

    explicit blacklist_entry(const std::string& s)
        : email(s)
        , created(0)
    {
    }

    bool operator<(const blacklist_entry& rh) const {
        return email < rh.email;
    }

    std::string email;
    std::time_t created;
};

struct _ListType {
    enum Enum {
        BLACK = 0,
        WHITE,
        ALL
    };
    template <typename Map>
    void fill(Map& m) const {
        m.insert(typename Map::value_type(BLACK, "black"));
        m.insert(typename Map::value_type(WHITE, "white"));
        m.insert(typename Map::value_type(ALL, "all"));
    }
    using Filler = _ListType;
};
using ListType = pgg::Enumeration<_ListType, _ListType::BLACK>;

struct blacklist {
    std::set<blacklist_entry> black, white;
};

using blacklist_ptr = boost::shared_ptr<blacklist>;

} // namespace furita::rules
