#pragma once

#include <string>
#include <stdexcept>
#include <boost/optional.hpp>
#include "rule.hpp"
#include <yamail/data/deserialization/json_reader.h>


namespace furita {
namespace rules_helpers {

enum AttachmentParamType {
    ALL                 =   0,
    WITH_ATTACHES       =   1,
    WITHOUT_ATTACHES    =   2
};

bool acceptableForExistsCondition(const std::string& field);
std::string findFieldType(const std::string &field);

class ConditionOper {
public:
    enum ConditionOperType {
        INVALID         = -1,

        MATCHES         = 1,
        NOT_MATCHES     = 2,
        CONTAINS        = 3,
        NOT_CONTAINS    = 4,
        EXISTS          = 5,
        NOT_EXISTS      = 6
    };

    ConditionOper(const std::string& _type) {
        if (_type == "1") {
            type = MATCHES;
        } else if (_type == "2") {
            type = NOT_MATCHES;
        } else if (_type == "3") {
            type = CONTAINS;
        } else if (_type == "4") {
            type = NOT_CONTAINS;
        } else if (_type == "5") {
            type = EXISTS;
        } else if (_type == "6") {
            type = NOT_EXISTS;
        } else {
            type = INVALID;
        }
    }

    bool valid() {
        return (type != INVALID);
    }

    bool negative() {
        return (type == NOT_MATCHES || type == NOT_CONTAINS || type == NOT_EXISTS);
    }

    rules::condition::oper_type toRuleOperType() {
        if (type == MATCHES || type == NOT_MATCHES) {
            return rules::condition::oper_type::MATCHES;
        } else if (type == CONTAINS || type == NOT_CONTAINS) {
            return rules::condition::oper_type::CONTAINS;
        } else {
            return rules::condition::oper_type::EXISTS;
        }
    }

private:
    ConditionOperType type;
};

class RuleType {
public:
    enum class RuleTypeType {
        USER            =   1,
        SYSTEM          =   3
    };

    static boost::optional<RuleType> create(const std::string& _type) {
        boost::optional<RuleType> res;
        if (_type == "user") {
            res = RuleType(RuleTypeType::USER);
        } else if (_type == "system") {
            res = RuleType(RuleTypeType::SYSTEM);
        } else if (!_type.empty()) {
            throw std::runtime_error("incorrect type: " + _type);
        }
        return res;
    }

    std::string toString() const {
        std::string res;
        switch (ruleType) {
            case RuleTypeType::USER:
                res = "user";
                break;
            case RuleTypeType::SYSTEM:
                res = "system";
                break;
        }
        return res;
    }

private:
    RuleType() {}
    RuleType(const RuleTypeType &_type) : ruleType(_type) {}

    RuleTypeType ruleType;
};

using OptionalRuleType = boost::optional<RuleType>;

struct Filter {
    std::string email;
    std::string type;
    std::string displayName;
    std::string fid;
};

using Filters = std::vector<Filter>;

// [MAILDEV-935] :: приоритет пользовательских правил всегда должен быть выше приоритета фильтров отписки
// [MAILDEV-1027] :: приоритет системных правил должен быть ещё больше, чем у пользовательских
// == Замечание ==
// правило считается более приоритетным, чем меньше значение priority
inline void hackListPriority(const rules::rule_list_ptr& rules) {
    boost::optional<int> maxPrioSystem;
    boost::optional<int> minPrioUser;

    for (const rules::rule_ptr& rule : *rules) {
        if (rule->type == "system") {
            maxPrioSystem = (!maxPrioSystem || *maxPrioSystem < rule->prio) ? rule->prio : *maxPrioSystem;
        } else if (rule->type == "user") {
            minPrioUser = (!minPrioUser || *minPrioUser > rule->prio) ? rule->prio : *minPrioUser;
        }
    }

    int systemDec = (maxPrioSystem && minPrioUser) ? abs(*maxPrioSystem - *minPrioUser) + 1 : 0;

    if (systemDec == 0) {
        return; // nothing to hack
    }

    for (const rules::rule_ptr& rule : *rules) {
        if (rule->type == "system") {
            rule->prio -= systemDec;
        }
    }
}

}   // namespace rules_helpers
}   // namespace furita

BOOST_FUSION_ADAPT_STRUCT(furita::rules_helpers::Filter,
    (std::string, email)
    (std::string, type)
    (std::string, displayName)
    (std::string, fid)
)
