#include <yplatform/find.h>
#include <furita/common/logger.h>
#include "util.h"
#include "buffer_chunk.h"
#include "bw_add_handler.h"

namespace furita {
namespace api {

struct BWAddHandler::Arguments {
    HttpStreamPtr stream;
    uint64_t uid;
    std::string email;
};

void BWAddHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    auto args = std::make_shared<Arguments>();
    try {
        const auto& params = stream->request()->url.params;
        args->stream = stream;
        args->uid = parameterValue<uint64_t>(params, "uid");
        args->email = parameterValue(params, "email");
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="blacklist_add operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    try {
        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(ctx));
        auto executor = pq->create_request_executor(ctx, args->uid, resolverFactory);
        auto futureResult = pq->get_all_lists(executor, args->uid);
        futureResult.add_callback(boost::bind(&BWAddHandler::handleList, this, args, executor, futureResult, ctx));
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="blacklist_add operation finished: status=error")
        return _handleError(stream, std::string("Internal server error"));
    }
}

void BWAddHandler::handleList(ArgumentsPtr args, pgg::RequestExecutor& executor, furita::pq::future<rules::blacklist_ptr> fbl, TContextPtr ctx) const {
    if (fbl.has_exception()) {
        return handleError(args->stream, "blacklist_add", fbl);
    }

    if (fbl.get()->black.count(rules::blacklist_entry(args->email)) ||
        fbl.get()->white.count(rules::blacklist_entry(args->email)))
    {
        return _handleError(args->stream, std::string("Already exists"));
    }
    auto pq = yplatform::find<pq::pq>("furita_pq");
    auto futureResult =
        pq->add_to_list(executor, args->uid, args->email, type.toString());
    futureResult.add_callback(boost::bind(&BWAddHandler::handleResult, this, args, futureResult, ctx));
}

void BWAddHandler::handleResult(ArgumentsPtr args, furita::pq::future<void> result, TContextPtr ctx) const {
    if (result.has_exception()) {
        return handleError(args->stream, "blacklist_add", result);
    }

    FURITA_LOG_NOTICE(ctx, logdog::message="blacklist_add operation finished: status=ok");

    boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);

    json_writer &writer = buffer->m_writer;
    writer.begin_object().add_member("session", args->stream->ctx()->uniq_id());
    writer.end_object().close();

    args->stream->set_code(ymod_webserver::codes::ok);
    args->stream->set_content_type("application", "json");
    args->stream->result_stream(writer.size())->send(buffer);
}

}   // namespace api
}   // namespace furita
