#include <yplatform/find.h>
#include <furita/common/logger.h>
#include "util.h"
#include "buffer_chunk.h"
#include "bw_handler.h"

namespace furita {
namespace api {

struct BWHandler::Arguments {
    HttpStreamPtr stream;

    uint64_t uid;
    sharpei::client::Mode mode;
    bool detailed_, master_;
};

void BWHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    auto args = std::make_shared<Arguments>();
    try {
        const auto& params = stream->request()->url.params;
        args->stream = stream;
        args->uid = parameterValue<uint64_t>(params, "uid");
        args->detailed_ = parameterValue(params, "detailed", false);
        args->master_ = parameterValue(params, "master", false);
        args->mode = args->master_
                           ? sharpei::client::Mode::WriteOnly
                           : sharpei::client::Mode::ReadWrite;
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="bw operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    try {
        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(ctx));
        auto executor = pq->create_request_executor(ctx, args->uid, resolverFactory, args->mode);

        furita::pq::future<rules::blacklist_ptr> futureResult;
        switch (type) {
            case rules::ListType::BLACK:
                futureResult = pq->get_list_by_type(executor, args->uid, type.toString());
                break;
            case rules::ListType::WHITE:
                futureResult = pq->get_list_by_type(executor, args->uid, type.toString());
                break;
            case rules::ListType::ALL:
                futureResult = pq->get_all_lists(executor, args->uid);
                break;
        }
        futureResult.add_callback(boost::bind(&BWHandler::handleResult, this, args, futureResult, ctx));
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="bw operation finished: status=error")
        return _handleError(stream, std::string("Internal server error"));
    }
}

namespace {

inline void write_entry(json_writer &w, const furita::rules::blacklist_entry &e) {
    w.begin_object();
    w.add_member("email", e.email);
    w.add_member("created", e.created);
    w.end_object();
}

} // namespace

void BWHandler::handleResult(ArgumentsPtr args, furita::pq::future<rules::blacklist_ptr> result, TContextPtr ctx) const {
    if (result.has_exception()) {
        return handleError(args->stream, "blacklist", result);
    }

    const std::set<rules::blacklist_entry> &bl = result.get()->black;
    const std::set<rules::blacklist_entry> &wl = result.get()->white;
    const std::set<rules::blacklist_entry> *addresses = NULL;

    if (type == rules::ListType::BLACK || type == rules::ListType::WHITE) {
        addresses = (type == rules::ListType::BLACK) ? &bl : &wl;
    }

    FURITA_LOG_NOTICE(ctx, logdog::message="blacklist operation finished: status=ok")

    boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);

    json_writer &writer = buffer->m_writer;
    writer.begin_object().add_member("session", args->stream->ctx()->uniq_id());

    if (addresses) {
        writer.begin_array("addresses");
        for (const auto& entry: *addresses) {
            writer.add_member(entry.email);
        }
        writer.end_array();
    } else {
        writer.begin_array("blacklist");
        for (const auto& entry: bl) {
            if (args->detailed_) {
                write_entry(writer, entry);
            } else {
                writer.add_member(entry.email);
            }
        }
        writer.end_array().begin_array("whitelist");
        for (const auto& entry: wl) {
            if (args->detailed_) {
                write_entry(writer, entry);
            } else {
                writer.add_member(entry.email);
            }
        }
        writer.end_array();
    }

    writer.end_object().close();

    args->stream->set_code(ymod_webserver::codes::ok);
    args->stream->set_content_type("application", "json");
    args->stream->result_stream(writer.size())->send(buffer);
}

}   // namespace api
}   // namespace furita
