#include <yplatform/find.h>
#include <furita/common/logger.h>
#include "util.h"
#include "buffer_chunk.h"
#include "bw_remove_handler.h"

namespace furita {
namespace api {

struct BWRemoveHandler::Arguments {
    HttpStreamPtr stream;
    uint64_t uid;
    std::vector<std::string> addrs_;
};

void BWRemoveHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    auto args = std::make_shared<Arguments>();
    try {
        const auto& params = stream->request()->url.params;
        args->stream = stream;
        args->uid = parameterValue<uint64_t>(params, "uid");

        auto emails = params.equal_range("email");
        if (emails.first == emails.second) {
            throw std::runtime_error("No enough parameters");
        }

        for (auto iter = emails.first; iter != emails.second; ++iter) {
            args->addrs_.push_back(iter->second);
        }
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="blacklist_remove operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    try {
        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(ctx));
        auto executor = pq->create_request_executor(ctx, args->uid, resolverFactory);

        auto futureResult = pq->
            remove_from_list(executor, args->uid, args->addrs_, type.toString());
        futureResult.add_callback(
            boost::bind(&BWRemoveHandler::handleResult, this, args, futureResult, ctx)
        );
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="blacklist_remove operation finished: status=error")
        return _handleError(stream, std::string("Internal server error"));
    }
}

void BWRemoveHandler::handleResult(ArgumentsPtr args, furita::pq::future<void> result, TContextPtr ctx) const {
    if (result.has_exception()) {
        return handleError(args->stream, "blacklist_remove", result);
    }

    FURITA_LOG_NOTICE(ctx, logdog::message="blacklist_remove operation finished: status=ok")

    boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);

    json_writer &writer = buffer->m_writer;
    writer.begin_object().add_member("session", args->stream->ctx()->uniq_id());
    writer.end_object().close();

    args->stream->set_code(ymod_webserver::codes::ok);
    args->stream->set_content_type("application", "json");
    args->stream->result_stream(writer.size())->send(buffer);
}

}   // namespace api
}   // namespace furita
