#pragma once

#include "common_handler.h"

#include <mail/furita/src/api/domain_rules_get/handler_impl.h>
#include <mail/furita/src/api/domain_rules_get/types.h>
#include <mail/furita/src/api/domain_rules_get/utils.h>
#include <mail/furita/src/api/util.h>

#include <furita/common/logger.h>

#include <yplatform/find.h>
#include <yplatform/reactor.h>

#include <ymod_httpclient/call.h>
#include <ymod_httpclient/cluster_client.h>

#include <memory>

namespace furita::api {

class TDomainRulesGetHandler : public CommonHandler, public std::enable_shared_from_this<TDomainRulesGetHandler>
{
public:
    void execute(HttpStreamPtr stream, TContextPtr ctx) const override {
        std::int64_t orgId = 0;
        const auto& params = stream->request()->url.params;
        try {
            orgId = parameterValue<std::int64_t>(params, "orgid");
        } catch (const std::exception& exc) {
            FURITA_LOG_ERROR(ctx, logdog::message="Exception while parsing orgid", logdog::exception=exc);
            return ResponseMessage(stream, ymod_webserver::codes::code::internal_server_error, exc.what());
        }

        auto& ioContext = *yplatform::global_reactor_set->get("global")->io();
        auto repository = yplatform::find<furita::Repository, std::shared_ptr>("furitadb");
        domain_rules_get::THandlerPtr handler = std::make_shared<domain_rules_get::THandlerImpl>(
            std::move(repository), ioContext);
        auto req = domain_rules_get::TRequest {
            .OrgId = orgId
        };

        namespace ph = std::placeholders;
        handler->Run(ctx,
            std::move(req),
            std::bind(&TDomainRulesGetHandler::handleRun, shared_from_this(), ph::_1, ph::_2, stream, ctx));
    }

private:
    void handleRun(boost::system::error_code ec, domain_rules_get::TResponse resp, HttpStreamPtr stream, TContextPtr ctx) const {
        if (ec) {
            FURITA_LOG_ERROR(ctx, logdog::error_code=ec, logdog::message="Error occured");
            return ResponseMessage(stream, ymod_webserver::codes::code::internal_server_error, "error occurred");
        }
        if (!resp.Rules) {
            return ResponseMessage(stream, ymod_webserver::codes::code::not_found, "rules not found");
        }
        if (!resp.Revision) {
            return ResponseMessage(stream, ymod_webserver::codes::code::not_found, "revision not found");
        }

        stream->set_code(ymod_webserver::codes::ok);
        stream->set_content_type("application", "json");

        auto respJson = domain_rules_get::utils::FromJson(*resp.Rules);
        respJson["revision"] = static_cast<std::int32_t>(*resp.Revision);

        stream->result_body(respJson.stringify());
    }
};

} // namespace furita::api
