#pragma once

#include <boost/system/error_code.hpp>

namespace furita::domain_rules_set {

enum class EError {
    Ok,
    ConditionQueriesCountError,
    EmailDoesntBelongToOrganization,
    LimitsViolation
};

} // namespace furita::domain_rules_set

namespace boost::system {

template <>
struct is_error_code_enum<furita::domain_rules_set::EError> : std::true_type {};

} // namespace boost::system

namespace furita::domain_rules_set {

class TErrorCategory final : public boost::system::error_category {

public:
    const char* name() const noexcept override {
        return "furita_domain_rules_set";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::Ok:
                return "Ok";
            case EError::ConditionQueriesCountError:
                return "Tupita responded with number of queries not equal to number of rules";
            case EError::EmailDoesntBelongToOrganization:
                return "Some email doesn't belong to the organization";
            case EError::LimitsViolation:
                return "Limits violation";
        }
        return "Unknown error code: " + std::to_string(value);
    }
};

inline const TErrorCategory& getErrorCategory() {
    static TErrorCategory errorCategory;
    return errorCategory;
}

inline auto make_error_code(EError ec) {
    return boost::system::error_code(static_cast<int>(ec), getErrorCategory());
}

} // namespace furita::domain_rules_set
