#include "handler_impl.h"
#include "error_code.h"

#include <mail/furita/src/api/common_handler.h>
#include <mail/furita/src/tupita/http_client.h>

#include <spdlog/details/format.h>

#include <boost/asio/coroutine.hpp>

#include <algorithm>

namespace furita::domain_rules_set {

using tupita::TTupitaRequest;
using blackbox::TBlackBoxRequest;
using blackbox::TBlackBoxResponse;

constexpr const size_t MAX_RULES_COUNT = 200;
constexpr const size_t MAX_CONDITION_LENGTH = 200000;
constexpr const size_t MAX_ACTIONS_COUNT = 30;

#include <boost/asio/yield.hpp>

void THandlerCoro::Run() {
    reenter(Coro) {
        FURITA_LOG_NOTICE(Ctx, logdog::message=fmt::format("Start setting rules for orgid {}", Req.OrgId));

        if (!CheckLimits()) {
            FURITA_LOG_ERROR(Ctx, logdog::message="Limits violation");
            return boost::asio::post(IoContext,
                std::bind(std::move(Callback), make_error_code(EError::LimitsViolation),
                    TResponse{.Message = "Limits violation"})
            );
        }

        yield CollectOrgIdsForEmailsFromActions();

        if (OrgIdError) {
            FURITA_LOG_ERROR(Ctx, logdog::error_code=OrgIdError, logdog::message="Error with orgid requests to blackbox");
            return boost::asio::post(IoContext, std::bind(std::move(Callback), OrgIdError, TResponse{.Message = "Error while resolving orgids for emails"}));
        }

        if (!CheckSameOrganization()) {
            FURITA_LOG_ERROR(Ctx, logdog::message="Some email doesn't belong to the organization");
            return boost::asio::post(IoContext,
                std::bind(std::move(Callback), make_error_code(EError::EmailDoesntBelongToOrganization),
                    TResponse{.Message = "Some email doesn't belong to the organization"})
            );
        }

        yield RequestTupita();

        if (TupitaErrorCode) {
            FURITA_LOG_ERROR(Ctx, logdog::error_code=TupitaErrorCode, logdog::message=TupitaResponse.ErrorMessage);
            return boost::asio::post(IoContext, std::bind(std::move(Callback), TupitaErrorCode, TResponse{.Message = TupitaResponse.ErrorMessage}));
        }

        if (TupitaResponse.Queries.size() != Req.Rules.size()) {
            auto errorCode = make_error_code(EError::ConditionQueriesCountError);
            FURITA_LOG_ERROR(Ctx,
                logdog::error_code=errorCode,
                logdog::message="Tupita responded with number of conditions not equal to number of rules"
            );
            return boost::asio::post(IoContext,
                std::bind(std::move(Callback),
                    errorCode,
                    TResponse{.Message = "Problem with number of conditions (not equal to rules number)"}
                )
            );
        }

        for (size_t i = 0; i < TupitaResponse.Queries.size(); ++i) {
            Req.Rules[i].ConditionQuery = TupitaResponse.Queries[i];
        }

        yield SaveToRepository();

        if (DbErrorCode) {
            FURITA_LOG_ERROR(Ctx, logdog::error_code=DbErrorCode, logdog::message="Database error occurred");
            return boost::asio::post(IoContext, 
                std::bind(std::move(Callback), DbErrorCode.base(), TResponse{.Message = "Database error occurred"})
            );
        }

        {
            std::string message = "Revision id: unknown";
            if (Revision) {
                message = fmt::format("Revision id: {}", *Revision);
            }
            FURITA_LOG_NOTICE(Ctx, logdog::message=message);
            boost::asio::post(IoContext, std::bind(std::move(Callback), boost::system::error_code{}, TResponse{.Message = std::move(message)}));
        }
    }
}

#include <boost/asio/unyield.hpp>

bool THandlerCoro::CheckLimits() {
    if (Req.Rules.size() > MAX_RULES_COUNT) {
        FURITA_LOG_NOTICE(Ctx, logdog::message=fmt::format("Rule count exceeds limit ({}, maximum {})", Req.Rules.size(), MAX_RULES_COUNT));
        return false;
    }

    for (const auto& rule : Req.Rules) {
        std::size_t conditionLength = rule.Condition.stringify().size();
        if (conditionLength > MAX_CONDITION_LENGTH) {
            FURITA_LOG_NOTICE(Ctx, logdog::message=fmt::format("Condition length exceeds limit ({}, maximum {})", conditionLength, MAX_CONDITION_LENGTH));
            return false;
        }

        std::size_t actionsCount = rule.Actions.size();
        if (actionsCount > MAX_ACTIONS_COUNT) {
            FURITA_LOG_NOTICE(Ctx, logdog::message=fmt::format("Action count exceeds limit ({}, maximum {})", actionsCount, MAX_ACTIONS_COUNT));
            return false;
        }
    }

    return true;
}

void THandlerCoro::CollectOrgIdsForEmailsFromActions() {
    std::vector<std::string> emails;

    for (const auto& rule : Req.Rules) {
        for (const auto& action : rule.Actions) {
            if (action.Email) {
                emails.push_back(*action.Email);
            }
        }
    }

    BlackBox->DoRequest(Ctx, TBlackBoxRequest{.Emails = std::move(emails)},
        [this, self = shared_from_this()](boost::system::error_code ec, TBlackBoxResponse resp) {
            OrgIdError = std::move(ec);
            EmailsOrgIds = std::move(resp.OrgIds);
            boost::asio::post(IoContext, std::bind(&THandlerCoro::Run, self));
        }
    );
}

bool THandlerCoro::CheckSameOrganization() {
    bool allInSameOrgId = std::all_of(EmailsOrgIds.begin(), EmailsOrgIds.end(),
        [orgIdStr = std::to_string(Req.OrgId)](const std::string& emailOrgId) {
            return orgIdStr == emailOrgId;
        }
    );
    return allInSameOrgId;
}

void THandlerCoro::RequestTupita() {
    TTupitaRequest tupitaReq {
        .OrgId = std::to_string(Req.OrgId)
    };
    for (const auto& rule : Req.Rules) {
        tupitaReq.Conditions.push_back(rule.Condition);
    }
    Tupita->DoRequest(Ctx, std::move(tupitaReq), [this, self = shared_from_this()](auto ec, auto response) {
        TupitaErrorCode = std::move(ec);
        TupitaResponse = std::move(response);
        boost::asio::post(IoContext, std::bind(&THandlerCoro::Run, self));
    });
}

void THandlerCoro::SaveToRepository() {
    SetDomainRulesParams params {
        .uniqId = UniqId{Ctx->uniq_id()},
        .orgId = OrgId{Req.OrgId},
        .rules = Rules{utils::RulesToJson(Req.Rules).stringify()}
    };

    Repository->setDomainRules(std::move(params),
        [this, self = shared_from_this()](auto ec, auto revision) {
            DbErrorCode = std::move(ec);
            Revision = std::move(revision);
            boost::asio::post(IoContext, std::bind(&THandlerCoro::Run, self));
        }
    );
}

} // namespace furita::domain_rules_set

