#pragma once

#include "interface.h"
#include "utils.h"
#include "types.h"

#include <mail/furita/src/tupita/http_client.h>
#include <mail/furita/src/blackbox/http_client.h>

#include <boost/asio/io_context.hpp>
#include <boost/system/error_code.hpp>
#include <boost/asio/coroutine.hpp>

#include <optional>
#include <vector>
#include <utility>
#include <memory>

namespace furita::domain_rules_set {

using tupita::TTupitaClientPtr;
using tupita::TTupitaResponse;
using blackbox::TBlackBoxClientPtr;

class THandlerCoro : public std::enable_shared_from_this<THandlerCoro> {
public:
    THandlerCoro(
        TContextPtr ctx,
        TRequest req,
        TCallback callback,
        RepositoryPtr repository,
        TTupitaClientPtr tupita,
        TBlackBoxClientPtr blackBox,
        boost::asio::io_context& ioContext)
    : Ctx(std::move(ctx))
    , Req(std::move(req))
    , Callback(std::move(callback))
    , Repository(std::move(repository))
    , Tupita(std::move(tupita))
    , BlackBox(std::move(blackBox))
    , IoContext(ioContext) {}

    void Run();

private:
    bool CheckLimits();
    void CollectOrgIdsForEmailsFromActions();
    bool CheckSameOrganization();
    void RequestTupita();
    void SaveToRepository();

private:
    TContextPtr Ctx;
    TRequest Req;
    TCallback Callback;
    RepositoryPtr Repository;
    TTupitaClientPtr Tupita;
    TBlackBoxClientPtr BlackBox;
    boost::asio::io_context& IoContext;

    boost::asio::coroutine Coro;

    boost::system::error_code OrgIdError;
    std::vector<std::string> EmailsOrgIds;

    mail_errors::error_code DbErrorCode;
    std::optional<Revision> Revision;

    boost::system::error_code TupitaErrorCode;
    TTupitaResponse TupitaResponse;
};

class THandlerImpl : public IHandler {
public:
    THandlerImpl(
        RepositoryPtr repository,
        TTupitaClientPtr tupita,
        TBlackBoxClientPtr blackBox,
        boost::asio::io_context& ioContext)
    : Repository(std::move(repository))
    , Tupita(std::move(tupita))
    , BlackBox(std::move(blackBox))
    , IoContext(ioContext) {}

    void Run(TContextPtr ctx, TRequest req, TCallback callback) override {
        auto coro = std::make_shared<THandlerCoro>(
            std::move(ctx),
            std::move(req),
            std::move(callback),
            Repository,
            Tupita,
            BlackBox,
            IoContext);
        boost::asio::post(IoContext, [coro = std::move(coro)] {
            coro->Run();
        });
    }

private:
    RepositoryPtr Repository;
    TTupitaClientPtr Tupita;
    TBlackBoxClientPtr BlackBox;
    boost::asio::io_context& IoContext;
};

} // namespace furita::domain_rules_set
