#include "utils.h"

#include <boost/algorithm/string.hpp>

#include <unordered_set>
#include <string>

namespace furita::domain_rules_set::utils {

yplatform::json_value FromJson(const std::string& json) {
    yplatform::json_value result;
    if (std::optional<std::string> error = result.parse(json); error) {
        throw std::invalid_argument("Invalid json: " + *error);
    }
    return result;
}

void ParseTerminal(const yplatform::json_value& val, TRule& rule) {
    if (val.isMember("terminal")) {
        if (!val["terminal"].is_bool()) {
            throw std::invalid_argument("Expected field \"terminal\" to be a boolean value");
        }
        rule.Terminal = val["terminal"].to_bool();
    }
}

void ParseScope(const yplatform::json_value& val, TRule& rule) {
    if (!val.isMember("scope")) {
        throw std::invalid_argument("Expected field \"scope\"");
    }
    if (!val["scope"].is_object()) {
        throw std::invalid_argument("Expected field \"scope\" to be an object)");
    }
    if (!val["scope"].isMember("direction") || !val["scope"]["direction"].is_string()) {
        throw std::invalid_argument("Expected field \"scope\" to have string member \"direction\"");
    }
    auto direction = val["scope"]["direction"].to_string();
    if (direction == "inbound") {
        rule.Scope.Direction = EDirectionScope::INBOUND;
    } else {
        throw std::invalid_argument("Unknown \"direction\" scope");
    }
}

void ParseCondition(const yplatform::json_value& val, TRule& rule) {
    if (!val.isMember("condition")) {
        throw std::invalid_argument("Expected field \"condition\"");
    }
    rule.Condition = val["condition"];
}

void ParseActions(const yplatform::json_value& val, TRule& rule) {
    if (!val.isMember("actions")) {
        throw std::invalid_argument("Expected field \"actions\"");
    }
    if (!val["actions"].is_array()) {
        throw std::invalid_argument("Expected field \"actions\" to be an array");
    }

    static const std::unordered_set<std::string> allowedValues{"forward", "drop"};
    const auto& actions = val["actions"];
    for (std::size_t i = 0; i < actions.size(); ++i) {
        const auto& action = actions[i];
        if (!action.is_object()) {
            throw std::invalid_argument("Expected action #" + std::to_string(i) + " to be an object");
        }
        if (!action.isMember("action") || !action["action"].is_string()) {
            throw std::invalid_argument("Expected action #" + std::to_string(i) + " to have string field \"action\"");
        }
        const auto& actionStr = action["action"].to_string();
        if (!allowedValues.contains(actionStr)) {
            throw std::invalid_argument("Expected field \"action\" #" + std::to_string(i) + 
                " to be one of the following values: " + boost::algorithm::join(allowedValues, ","));
        }

        if (actionStr == "forward") {
            if (!action.isMember("data")) {
                throw std::invalid_argument("Expected action #" + std::to_string(i) + " to have string field \"data\"");
            }
            const auto& data = action["data"];
            if (!data.isMember("email") || !data["email"].is_string()) {
                throw std::invalid_argument("Expected action #" + std::to_string(i) + " to have string field \"data\".\"email\"");
            }

            rule.Actions.push_back(TAction{.Action = actionStr, .Email = data["email"].to_string()});
        } else if (actionStr == "drop") {
            rule.Actions.push_back(TAction{.Action = actionStr});
        }
    }
}

TRule ParseRule(const yplatform::json_value& val) {
    TRule rule;

    ParseTerminal(val, rule);
    ParseScope(val, rule);
    ParseCondition(val, rule);
    ParseActions(val, rule);

    return rule;
}

std::vector<TRule> ParseRules(const yplatform::json_value& val) {
    if (!val.isMember("rules")) {
        throw std::invalid_argument("Expected field \"rules\"");
    }
    if (!val["rules"].is_array()) {
        throw std::invalid_argument("Expected field \"rules\" to be an array");
    }
    std::vector<TRule> result;
    const auto& rules = val["rules"];
    for (auto it = rules.array_begin(); it != rules.array_end(); ++it) {
        result.push_back(ParseRule(*it));
    }
    return result;
}

TRulesSpec ParseRulesSpec(const std::string& body) {
    auto val = FromJson(body); 
    TRulesSpec result;
    result.Rules = ParseRules(val);
    return result;
}

yplatform::json_value RuleToJson(const TRule& rule) {
    yplatform::json_value result;

    if (rule.Terminal) {
        result["terminal"] = *rule.Terminal;
    }
    result["scope"].set_object();
    switch (rule.Scope.Direction) {
        case EDirectionScope::INBOUND:
            result["scope"]["direction"] = "inbound";
            break;
        default:
            break;
    }

    result["condition"] = rule.Condition;
    result["condition_query"] = rule.ConditionQuery;
    result["actions"].set_array();
    for (const auto& action : rule.Actions) {
        yplatform::json_value actionObj;
        actionObj["action"] = action.Action;
        if (action.Email) {
            actionObj["data"]["email"] = *action.Email;
        }
        result["actions"].push_back(std::move(actionObj));
    }
    return result;
}

yplatform::json_value RulesToJson(const std::vector<TRule>& rules) {
    yplatform::json_value convertedRules;
    for (const auto& rule : rules) {
        convertedRules.push_back(RuleToJson(rule));
    }
    yplatform::json_value res;
    res["rules"] = std::move(convertedRules);
    return res;
}

}
