#pragma once

#include "common_handler.h"

#include <mail/furita/src/api/domain_rules_set/handler_impl.h>
#include <mail/furita/src/api/domain_rules_set/types.h>
#include <mail/furita/src/api/domain_rules_set/utils.h>
#include <mail/furita/src/api/util.h>

#include <furita/common/logger.h>

#include <yplatform/find.h>
#include <yplatform/reactor.h>

#include <ymod_httpclient/call.h>
#include <ymod_httpclient/cluster_client.h>

#include <spdlog/details/format.h>

#include <memory>

namespace furita::api {

constexpr const size_t MAX_BODY_LENGTH = 1 * 1024 * 1024;

class TDomainRulesSetHandler : public CommonHandler, public std::enable_shared_from_this<TDomainRulesSetHandler>
{
public:
    void execute(HttpStreamPtr stream, TContextPtr ctx) const override {
        auto body = boost::copy_range<std::string>(stream->request()->raw_body);
        if (body.size() > MAX_BODY_LENGTH) {
            FURITA_LOG_ERROR(ctx, logdog::message=fmt::format("Too long body (size={}, maximum={})", body.size(), MAX_BODY_LENGTH));
            return ResponseMessage(stream, ymod_webserver::codes::code::internal_server_error, "Too long body");
        }

        domain_rules_set::TRulesSpec rulesSpec;
        try {
            rulesSpec = domain_rules_set::utils::ParseRulesSpec(body);
        } catch (const std::exception& exc) {
            FURITA_LOG_ERROR(ctx, logdog::message="Exception while parsing json", logdog::exception=exc);
            return ResponseMessage(stream, ymod_webserver::codes::code::internal_server_error, exc.what());
        }

        std::int64_t orgId = 0;
        const auto& params = stream->request()->url.params;
        try {
            orgId = parameterValue<std::int64_t>(params, "orgid");
        } catch (const std::exception& exc) {
            FURITA_LOG_ERROR(ctx, logdog::message="Exception while parsing orgid", logdog::exception=exc);
            return ResponseMessage(stream, ymod_webserver::codes::code::internal_server_error, exc.what());
        }

        auto& ioContext = *yplatform::global_reactor_set->get("global")->io();
        auto repository = yplatform::find<furita::Repository, std::shared_ptr>("furitadb");
        auto tvmModule = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("tvm");

        auto tupitaHttpClient = yplatform::find<ymod_httpclient::cluster_call, std::shared_ptr>("tupita_http_client");
        tupita::TTupitaClientPtr tupitaClient = std::make_shared<tupita::TTupitaClientImpl>(std::move(tupitaHttpClient), std::move(tvmModule), ioContext);

        auto blackboxHttpClient = yplatform::find<ymod_httpclient::cluster_call, std::shared_ptr>("blackbox_http_client");
        blackbox::TBlackBoxClientPtr blackBoxClient = std::make_shared<blackbox::TBlackBoxClient>(std::move(blackboxHttpClient), ioContext);

        domain_rules_set::THandlerPtr handler = std::make_shared<domain_rules_set::THandlerImpl>(
            std::move(repository), std::move(tupitaClient), std::move(blackBoxClient), ioContext);

        auto req = domain_rules_set::TRequest {
            .OrgId = orgId,
            .Rules = rulesSpec.Rules
        };

        namespace ph = std::placeholders;
        handler->Run(ctx,
            std::move(req),
            std::bind(&TDomainRulesSetHandler::handleRun, shared_from_this(), ph::_1, ph::_2, stream, ctx));
    }

private:
    void handleRun(boost::system::error_code ec, domain_rules_set::TResponse resp, HttpStreamPtr stream, TContextPtr ctx) const {
        if (ec) {
            FURITA_LOG_ERROR(ctx, logdog::error_code=ec, logdog::message="Error occured");
            stream->set_code(ymod_webserver::codes::internal_server_error);
            return ResponseMessage(stream, ymod_webserver::codes::internal_server_error, resp.Message);
        }
        ResponseMessage(stream, ymod_webserver::codes::ok, resp.Message);
    }
};

} // namespace furita::api
