#include <yplatform/find.h>
#include "util.h"
#include "buffer_chunk.h"
#include "edit_handler.h"
#include <furita/common/logger.h>
#include <furita/common/rule_helper.hpp>
#include <ymod_httpclient/cluster_client.h>
#include <butil/email/helpers.h>

namespace furita {
namespace api {

namespace
{
    rules::action_ptr make_label_action(const std::string &label)
    {
        if (label == "lid_read")
            return boost::make_shared<rules::action>("status", "RO");
        if (label == "lid_unread")
            return boost::make_shared<rules::action>("status", "New");
        return boost::make_shared<rules::action>("movel", label);
    }
}

void EditHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    TRuleParams rule_params;
    std::string letter;
    auto rule = boost::make_shared<rules::rule>();
    try {
        auto& params = stream->request()->url.params;
        rule_params.uid = parameterValue<uint64_t>(params, "uid");
        std::string id_str = parameterValue(params, "id", std::string());
        if (!id_str.empty()) {
            rule_params.id = std::stoull(id_str);
        }
        letter = parameterValue(params, "letter", std::string("nospam")); // all, nospam, clearspam
        rules::condition::link_type logic = parameterValue<int>(params, "logic", rules::condition::link_type::OR)
            ? rules::condition::link_type::AND
            : rules::condition::link_type::OR; // 0 - OR, 1 - AND
        int attachment = parameterValue<int>(params, "attachment", rules_helpers::AttachmentParamType::ALL); // 0 - all, 1 - with attaches, 2 - no attaches
        rule_params.last = parameterValue(params, "order", false);

        rule->name = parameterValue(params, "name");
        rule->stop = parameterValue(params, "stop", false);

        if (!letter.empty()
            && (letter == "all" || letter == "nospam" || letter == "clearspam"))
        {
            rule->conditions->push_back(boost::make_shared<rules::condition>(
                letter, "", rules::condition::link_type::AND, rules::condition::oper_type::MATCHES,
                false));
        }

        if (attachment != rules_helpers::AttachmentParamType::ALL)
        {
            rule->conditions->push_back(boost::make_shared<rules::condition>(
                "att", "", rules::condition::link_type::AND, rules::condition::oper_type::MATCHES,
                (attachment != rules_helpers::AttachmentParamType::WITH_ATTACHES)));
        }

        auto divs = params.equal_range("field1");
        auto operations = params.equal_range("field2");
        auto patterns = params.equal_range("field3");
        for (auto iterDiv = divs.first, iterOp = operations.first, iterPattern = patterns.first;
            iterDiv != divs.second && iterOp != operations.second; ++iterDiv, ++iterOp)
        {
            const auto& div = iterDiv->second;
            rules_helpers::ConditionOper op(iterOp->second);

            if (div.empty() || !op.valid()) {
                continue;
            }

            if (div != "virus" &&
                div != "spam" &&
                div != "subscribes" &&
                div != "list" &&
                div != "att" &&
                op.toRuleOperType() != rules::condition::oper_type::EXISTS)
            {
                if (iterPattern == patterns.second) {
                    continue;
                }
                if (iterPattern->second.empty()) {
                    ++iterPattern;
                    continue;
                }
            }

            // we can create EXISTS condition only for headers
            if (op.toRuleOperType() == rules::condition::oper_type::EXISTS &&
                !rules_helpers::acceptableForExistsCondition(div)) {
                continue;
            }

            // we can't use any other operations except MATCHES for message's type
            if (div == "type" && op.toRuleOperType() != rules::condition::oper_type::MATCHES) {
                continue;
            }

            const std::string pattern = (iterPattern != patterns.second) ? (iterPattern++)->second : "";

            rule->conditions->push_back(boost::make_shared<rules::condition>(
                div,
                pattern,
                logic,
                op.toRuleOperType(),
                op.negative())
            );
        }

        if (rule->conditions->empty()) {
            throw std::runtime_error("Can't create rule without conditions");
        }

        bool move = false, label = false, forward = false, reply = false, notify = false;
        auto clickers = params.equal_range("clicker");
        for (auto iterClick = clickers.first; iterClick != clickers.second; ++iterClick) {
            if (!move && iterClick->second == "move") {
                const std::string &fid = parameterValue(params, "move_folder");
                if (fid.empty()) {
                    continue;
                }
                rule->actions->push_back(boost::make_shared<rules::action>("move", fid));
                move = true;
                continue;
            }

            if (!move && iterClick->second == "delete") {
                rule->actions->push_back(boost::make_shared<rules::action>("delete", ""));
                move = true;
                continue;
            }

            if (!label && iterClick->second == "movel") {
                auto labels = params.equal_range("move_label");
                for (auto iterLabel = labels.first; iterLabel != labels.second; ++iterLabel) {
                    if (!iterLabel->second.empty()) {
                        rule->actions->push_back(make_label_action(iterLabel->second));
                    }
                }
                label = (labels.first != labels.second);
                continue;
            }

            if (!forward && (iterClick->second == "forward" || iterClick->second == "forwardwithstore")) {
                if (letter == "clearspam" || letter == "all") {
                    throw std::runtime_error("Can't forward spam");
                }

                auto iterFwdAddr = params.find("forward_address");
                if (iterFwdAddr == params.end()) {
                    throw std::runtime_error("parameter not found: forward_address");
                }

                try {
                    const std::string email = EmailHelpers::idnaize(iterFwdAddr->second);
                    if (!check_address(email)) {
                        throw std::runtime_error({});
                    }
                    rule->actions->push_back(
                        boost::make_shared<rules::action>(iterClick->second, email)
                    );
                } catch (const std::runtime_error&) {
                    throw std::runtime_error("bad address: forward_address");
                }

                forward = true;
                continue;
            }

            if (!reply && iterClick->second == "reply") {
                const std::string &ans = parameterValue(params, "autoanswer");
                if (ans.empty()) {
                    continue;
                }
                rule->actions->push_back(boost::make_shared<rules::action>("reply", ans));
                reply = true;
                continue;
            }

            if (!notify && iterClick->second == "notify") {
                auto addresses = params.equal_range("notify_address");
                for (auto iterAddr = addresses.first; iterAddr != addresses.second; ++iterAddr) {
                    try {
                        const std::string email = EmailHelpers::idnaize(iterAddr->second);
                        if (check_address(email)) {
                            rule->actions->push_back(
                                boost::make_shared<rules::action>("notify", email)
                            );
                        } else {
                            throw std::runtime_error({});
                        }
                    } catch (const std::runtime_error&) {
                        throw std::runtime_error("bad address: notify_address");
                    }
                }
                notify = true;
                continue;
            }
        }

        if (rule->actions->empty()) {
            throw std::runtime_error("Can't create rule without actions");
        }

        if ((forward || notify) && !parameterValue(params, "noconfirm", false)) {
            rule_params.from = parameterValue(params, "from");
            rule_params.lang = parameterValue(params, "lang");
            rule_params.confirm_domain = parameterValue(params, "confirm_domain");
            rule_params.auth_domain = parameterValue(params, "auth_domain");
        }

        rule_params.rule = rule;
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="edit operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    try {
        auto processor = yplatform::find<furita::processor::impl, std::shared_ptr>("furita_processor");
        const auto& so_check_options = processor->get_so_check_options();

        if (so_check_options.use) {
            auto& headers = stream->request()->headers;
            auto iter = headers.find("x-real-ip");
            std::string user_ip = iter == headers.end() ? "" : iter->second;

            auto cluster_client = yplatform::find<yhttp::cluster_client, std::shared_ptr>("so_check_client");
            processor::so_check::TParams so_check_params{user_ip, std::to_string(rule_params.uid), so_check_options.form_realpath,
                                                         so_check_options.url};
            auto future_so_check_result = processor::so_check::so_check_filter(ctx, rule, cluster_client, so_check_params);
            future_so_check_result.add_callback(boost::bind(&EditHandler::handleSoCheckResult, this,
                                                            stream, future_so_check_result, so_check_options.dry_run, rule_params, ctx));
        } else {
            ruleEdit(stream, rule_params, ctx);
        }
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="edit operation finished: status=error")
        return _handleError(stream, std::string("Internal server error"));
    }
}

void EditHandler::ruleEdit(HttpStreamPtr stream, const TRuleParams& rule_params, TContextPtr ctx) const {
    auto processor = yplatform::find<processor::processor>("furita_processor");
    auto futureResult = processor->rule_edit(ctx, rule_params.uid, rule_params.id, rule_params.rule, rule_params.last,
                                             rule_params.from, rule_params.lang, rule_params.confirm_domain, rule_params.auth_domain);
    futureResult.add_callback(boost::bind(&EditHandler::handleResult, this, stream, futureResult, ctx));
}

void EditHandler::handleSoCheckResult(HttpStreamPtr stream, future<processor::so_check::EResult> future_so_check_result,
                                      bool dry_run, const TRuleParams& rule_params, TContextPtr ctx) const {
    const auto so_check_result = future_so_check_result.get();
    switch (so_check_result) {
        case processor::so_check::EResult::SPAM:
            FURITA_LOG_NOTICE(ctx, logdog::message="edit operation: status=error, report='SO: spam filter'")
            if (!dry_run) {
                return _handleError(stream, std::string("Bad filter"), ymod_webserver::codes::code::forbidden);
            }
            break;

        case processor::so_check::EResult::ERROR:
            FURITA_LOG_ERROR(ctx, logdog::message="edit operation: status=error, report='SO: error'")
            if (!dry_run) {
                return _handleError(stream, std::string("Internal server error"));
            }
            break;

        case processor::so_check::EResult::OK:
            FURITA_LOG_DEBUG(ctx, logdog::message="edit operation: status=ok, report='SO: OK'")
    }

    ruleEdit(stream, rule_params, ctx);
}

void EditHandler::handleResult(HttpStreamPtr stream, future<std::pair<processor::status_result, boost::optional<uint64_t>>> result, TContextPtr ctx) const {
        try {
            auto result_edit = result.get();

            auto status = result_edit.first;
            auto data = result_edit.second;

            if(status == processor::status_result::FAIL) {
                std::string description = "сreate rule is prohibited";
                FURITA_LOG_ERROR(ctx, logdog::message="edit operation finished: status=error, report='" + description + "'")

                return handleFail(stream, description);

            } else if(status == processor::status_result::SUCCESS) {
                FURITA_LOG_NOTICE(ctx, logdog::message="edit operation finished: status=ok, id=" + std::to_string(data.value()))

                boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);
                json_writer &writer = buffer->m_writer;
                writer.begin_object().add_member("session", stream->ctx()->uniq_id());
                writer.add_member("id", std::to_string(*data));
                writer.end_object().close();

                stream->set_code(ymod_webserver::codes::ok);
                stream->set_content_type("application", "json");
                stream->result_stream(writer.size())->send(buffer);
            }
        } catch(const std::exception &e) {
            FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="edit operation finished: status=error")
            return _handleError(stream, e.what());
        }
}

}   // namespace api
}   // namespace furita
