#include <yplatform/find.h>
#include <furita/common/logger.h>
#include <furita/pq/pq.hpp>
#include "util.h"
#include "buffer_chunk.h"
#include "enable_handler.h"

using namespace furita::api;

struct EnableHandler::Arguments {
    HttpStreamPtr stream;
    uint64_t uid_, id_;
    bool enabled_;
};

void EnableHandler::execute(HttpStreamPtr stream, TContextPtr ctx) const {
    auto args = std::make_shared<Arguments>();
    try {
        const auto& params = stream->request()->url.params;
        args->stream = stream;
        args->uid_ = parameterValue<uint64_t>(params, "uid");
        args->id_ = parameterValue<uint64_t>(params, "id");
        args->enabled_ = parameterValue(params, "enabled", true);
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="enable operation finished: status=error")
        return handleFail(stream, "No enough parameters");
    }

    try {
        auto pq = yplatform::find<pq::pq>("furita_pq");
        auto resolverFactory = pgg::createSharpeiUidResolverFactory(pq->create_sharpei_params(ctx));
        auto executor = pq->create_request_executor(ctx, args->uid_, resolverFactory);

        auto futureResult = pq->
            enable_rule(executor, args->uid_, args->id_, args->enabled_);
        futureResult.add_callback(boost::bind(&EnableHandler::handleResult, this, args, futureResult, ctx));
    } catch (const std::exception &e) {
        FURITA_LOG_ERROR(ctx, logdog::exception=e, logdog::message="enable operation finished: status=error")
        return _handleError(stream, std::string("Internal server error"));
    }
}

void EnableHandler::handleResult(ArgumentsPtr args, future<void> result, TContextPtr ctx) const {
    if (result.has_exception()) {
        return handleError(args->stream, "enable", result);
    }

    FURITA_LOG_NOTICE(ctx, logdog::message="enable operation finished: status=ok")

    boost::shared_ptr<json_buffer_chunk> buffer(new json_buffer_chunk);

    json_writer &writer = buffer->m_writer;
    writer.begin_object().add_member("session", args->stream->ctx()->uniq_id());
    writer.end_object().close();

    args->stream->set_code(ymod_webserver::codes::ok);
    args->stream->set_content_type("application", "json");
    args->stream->result_stream(writer.size())->send(buffer);
}

